/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XBUTTON.H                                     */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of QubeOS                                  */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XBUTTON_H_INCLUDED__
#define __XBUTTON_H_INCLUDED__

#include<matrixos.h>

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

/*
  Button palette. It's set in button initializing function as the palette
  of the button. This settings are only default. The real settings are
  read from registers when the xbutton library is load into the memory.

  You can later get these values from your view ( in this case from the
  button ) by calling my_view->get_color(my_view, @index) where index
  is the index of these values. E.g. my_button->get_color(my_button, 0)
  is CO_SYSx20.

  Note that these values are not used. There is [colors] section in
  registers, from where are colors read. See "but_ini"
  at the bottom of this file for more information.
*/
extern l_color  pal_button[];


/* button flags, can use combination of followings */

/*
 normal button
*/
#define BF_NORMAL        0x00

/*
 button is arrounded by black rectangle and when enter is pressed, it calls the
 message of this button
*/
#define BF_DEFAULT       0x01

/*
 Draws the easy button without patterns
*/
#define BF_EASY          0x02


/* t_button */

#define BUTTON_ON_PRESS_FUNC (l_bool(*)(p_button, l_dword))

typedef struct t_button  *p_button;
typedef struct t_button {

/**
 * Inherited functions / variabes from t_view and t_object objects
*/
  struct t_view    obclass;

/**
 * Flags of the button. see BF_XXXX above.
*/
  l_int     flags;

/**
 * Caption of the button.
*/
  l_text    caption;

  l_dword   on_press_arg;

/**
 * Draws the state of the button. If @press is TRUE the button is drawn
 * as pressed one, otherwise like the normal.
 *
 * TODO : Drawing without patterns
*/
  void     (*draw_state) ( p_button o, l_int press );


/**
 * Tests if the button is default. When the enter is pressed from the
 * dialog, this button is pressed.
 *
*/
  l_bool   (*is_default) ( p_button o );


/**
 * Function is set in the button_init,button_new function and points
 * to a function that will be called whenever button was pressed.
*/
  l_bool   (*on_press) ( p_button o, l_dword on_press_arg );


  p_button (*get_default) ( p_button o );

} t_button;

#define BUTTON(o)   ((p_button)(o))

extern p_button def_button;

#define DEF_BUTTON_FUNC(y) DEF_FUNC(def_button,y)
#define DEF_BUTTON_VIEW_FUNC(y) DEF_FUNC(VIEW(def_button),y)
#define DEF_BUTTON_VIEW_OBJECT_FUNC(y) DEF_FUNC(OBJECT(def_button),y)



/**
 * Creates the new button in the area @r relative to the parent.
 *
 * @caption - the caption of the button.
 * @flags   - flags of the button, can be one of following :
 *
 *            BF_NORMAL  - normal button
 *            BF_DEFAULT - button is defualt. If the eneter is pressed,
 *                         this is the first that will be called.
 *
 * @on_press - the function that will be called if the button was pressed.
 * @on_press_arg - is the argument to be inserted as the 2nd one, when
 *                 function @on_press is called.
*/

extern p_button  (*button_init) ( p_button o, t_rect r, l_text caption, l_int flags, l_bool (*on_press)(p_button o, l_dword on_press_arg), l_dword on_press_arg );
p_button  _button_init ( p_button o, t_rect r, l_text caption, l_int flags, l_bool (*on_press)(p_button o, l_dword on_press_arg), l_dword on_press_arg );


/**
 * The same like button_init, but by this function couldn't use the
 * 1st argument and the height of the button is automatically stretched
 * to the success height. The same as :
 *
 * r.b.y = r.a.y+low_grx_width(pat_button_center);
 *
 * return button_init(_malloc(sizeof(t_button)), r, caption, flags, on_press, NULL);
 *
 * Note :
 * Use this function in end-level programming and function _button_init in
 * API library when you will need to make inherited object from the button.
*/
p_button  button_new ( t_rect r, l_text caption, l_int flags, l_bool (*on_press)(p_button o, l_dword on_press_arg), l_dword on_press_arg );


#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xbutton.h */

