/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XSELECT.H                                     */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of QubeOS                                  */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XSELECT_H_INCLUDED__
#define __XSELECT_H_INCLUDED__

#include<matrixos.h>

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif


/*
  Select palette. It's set in button initializing function as the palette
  of the select. This settings are only default. The real settings are
  read from registers when the xselect library is load into the memory.

  You can later get these values from your view ( in this case from the
  select ) by calling my_view->get_color(my_view, @index) where index
  is the index of these values. E.g. my_select->get_color(my_select, 0)
  is CO_SYSx20.

  Note that these values are not used. There is [colors] section in
  registers, from where are colors read. See "select_ini"
  at the bottom of this file for more information.
*/
extern l_color  pal_select[];

/*
  SelectGroup palette. It's set in button initializing function as the palette
  of the selectgroup. This settings are only default. The real settings are
  read from registers when the xselect library is load into the memory.

  You can later get these values from your view ( in this case from the
  select ) by calling my_view->get_color(my_view, @index) where index
  is the index of these values. E.g. my_selectgroup->get_color(my_select, 0)
  is CO_SYSx20.

  Note that these values are not used. There is [colors] section in
  registers, from where are colors read. See "select_ini"
  at the bottom of this file for more information.
*/
extern l_color  pal_selectgroup[];


/**
 * t_selectgroup alignments
*/
#define SGA_TOPLEFT        0
#define SGA_TOPRIGHT       1
#define SGA_BOTTOMLEFT     2
#define SGA_BOTTOMRIGHT    3

/**
 * t_select flags
*/
/*
 * Used when more items can be selected in
 * t_select object at one time
*/
#define SELF_MORESEL    0x0001


typedef struct t_select_item *p_select_item;
typedef struct t_select_item {

  /* Name of the item */
  l_text       id_name;
  /* Text for the item */
  l_text       caption;
  /* Flag to know if the item is selected */
  l_bool       select;
  /* Flag to know is the item is enabled or disabled. */
  l_bool       enable;
  /* Pointer to anywhere */
  l_dword      ptr;
  /* Bounds of item */
  t_rect       bounds;

  l_char       reserved[32];

} t_select_item;


/**
 * t_select flags
*/

typedef struct t_select *p_select;
typedef struct t_select {

  struct t_view   obclass;

  p_vector         items;
  l_int            flags;
  l_int            focus;
  l_int            focus_keyb;

  p_select_item  (*get_select_item_one) ( p_select o );
  p_select_item  (*get_item) ( p_select o, l_text id_name );
  void           (*add_item) ( p_select o, p_select_item item );
  void           (*set_item_enable) ( p_select o, l_text id_name, l_bool enable );
  void           (*set_item_ptr) ( p_select o, l_text id_name, l_dword ptr );
  void           (*set_item_select) ( p_select o, l_text id_name, l_bool select );
  p_select_item  (*get_item_under_mouse) ( p_select o, t_point where );
  void           (*draw_item) ( p_select o, p_select_item item );
  p_select_item  (*get_item_focus) ( p_select o );
  p_select_item  (*get_item_focus_keyb) ( p_select o );
  void           (*item_select) ( p_select o, p_select_item i, l_bool select );

} t_select;

#define SELECT(o)   ((p_select)(o))

extern p_select def_select;

#define DEF_SELECT_FUNC(y) DEF_FUNC(def_select,y)
#define DEF_SELECT_VIEW_FUNC(y) DEF_FUNC(VIEW(def_select),y)
#define DEF_SELECT_VIEW_OBJECT_FUNC(y) DEF_FUNC(OBJECT(def_select),y)


typedef struct t_selectgroup *p_selectgroup;
typedef struct t_selectgroup {

  struct t_view   obclass;

  l_text          caption;
  l_int           alignc;

} t_selectgroup;

#define SELECTGROUP(o)   ((p_selectgroup)(o))

extern p_selectgroup def_selectgroup;

#define DEF_SELECTGROUP_FUNC(y) DEF_FUNC(def_selectgroup,y)
#define DEF_SELECTGROUP_VIEW_FUNC(y) DEF_FUNC(VIEW(def_selectgroup),y)
#define DEF_SELECTGROUP_VIEW_OBJECT_FUNC(y) DEF_FUNC(OBJECT(def_selectgroup),y)


/**
 * Makes new select item in the rect @r
*/
p_select_item  select_item_new ( t_rect r, l_text id_name, l_text caption, l_bool select, l_dword ptr );


/**
 * Release memory of item
*/
void    select_item_free ( l_ptr ptr );


/**
 * Returns TRUE if items is enabled or FALSE if not.
*/
l_bool  select_item_get_enable ( p_select_item o );


/**
 * Returns TRUE if items is selected FALSE if not.
*/
l_bool  select_item_get_selected ( p_select_item o );


/**
 * Returns pointer of the selected item
*/
l_dword   select_item_get_ptr ( p_select_item o );


/**
 * Initializes new select object that's usefull as the parent class for
 * checkboxes or radio buttons. If the @flags is set to SELF_MORESEL than
 * it's able to select more than one item in the select at one time. It's
 * used in checkboxes.
*/
extern p_select  (*select_init) ( p_select o, t_rect r, l_int flags );
p_select  _select_init ( p_select o, t_rect r, l_int flags );


/**
 * Makes new select object that's usefull as the parent class for
 * checkboxes or radio buttons. If the @flags is set to SELF_MORESEL than
 * it's able to select more than one item in the select at one time. It's
 * used in checkboxes.
*/
p_select  select_new ( t_rect r, l_int flags );



/**
 * Initializes new static frame that isolates group, where @caption is
 * caption of the group and @align can be one of the following values :
 *
 * SGA_TOPLEFT   - default one
 * SGA_TOPRIGHT
 * SGA_BOTTOMLEFT
 * SGA_BOTTOMRIGHT
*/
extern p_selectgroup  (*selectgroup_init) ( p_selectgroup o, t_rect r, l_text caption, l_int align );
p_selectgroup  _selectgroup_init ( p_selectgroup o, t_rect r, l_text caption, l_int align );

/**
 * Makes new static frame that isolates group, where @caption is
 * caption of the group and @align can be one of the following values :
 *
 * SGA_TOPLEFT   - default one
 * SGA_TOPRIGHT
 * SGA_BOTTOMLEFT
 * SGA_BOTTOMRIGHT
*/
p_selectgroup  selectgroup_new ( t_rect r, l_text caption, l_int align );

#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xselect.h */

