/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XTXTLINE.H                                    */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of QubeOS                                  */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XTXTLINE_H_INCLUDED__
#define __XTXTLINE_H_INCLUDED__

#include<matrixos.h>

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif


/**
 * Textline flags
*/

/* None flags */
#define TF_NONE           0x0000
/* Text in t_textline can't be rewriten. */
#define TF_REWRITEUNABLE  0x00001
/* t_textline will control TAB key. */
#define TF_TABCTRL        0x00002

/**
 * Textline cursor flags
*/
/* Cursor is visible */
#define  TEXTLINE_CUR_VISIBLE     0x0001
/* Internal flag */
#define  TEXTLINE_CUR_VISIBLE2    0x0002
/* Ticking of cursor */
#define  TEXTLINE_CUR_VISIBLETICK 0x0004

/*
  Textline palette. It's set in textline initializing function as the palette
  of the textline. This settings are only default. The real settings are
  read from registers when the xtxtline library is load into the memory.

  You can later get these values from your view ( in this case from the
  textline ) by calling my_view->get_color(my_view, @index) where index
  is the index of these values. E.g. my_textline->get_color(my_textline, 0)
  is CO_SYSx20.

  Note that these values are not used. There is [colors] section in
  registers, from where are colors read. See "but_ini"
  at the bottom of this file for more information.
*/
extern l_color  pal_textline[];


typedef struct t_textline *p_textline;
typedef struct t_textline {

/**
 * Inherited functions / variabes from t_view and t_object objects
*/
  struct t_view  obclass;

/**
 * Flags of the cursor - see TEXTLINE_CUR_XXXX.
 * Keeps only internal information.
*/
  l_int          cursor_flags;

/**
 * The last position of the cursor.
*/
  t_point        cursor_last_pos;

/**
 * Flags of the textline, see TF_XXXX above
*/
  l_int          flags;

/**
 * Data of the textline ended by ZERO.
*/
  l_text         text;

/**
 * The current position of the cursor.
*/
  l_int          pos;

/**
 * The first visible character in the textline. This information is the
 * position of scrolling. If this number is 5, it means we doesn't see
 * 5 characters on the left side that were scrolled.
*/
  l_int          line;

/**
 * Maximum number of characters that may be inserted to the textline
*/
  l_int          limit;

/**
 * The begining of the selection. If the value is lower than ZERO,
 * there is no selection inside the textline.
*/
  l_int          sel_from;

/**
 * The end of selection. If the value is lower than ZERP,
 * there is no selection inside the textline.
*/
  l_int          sel_to;

  t_point        sel_safe;

/**
 * For internal use only.
 * There is stored the information from where we start to select the text.
*/
  l_int          sel_first_from;
  l_int          sel_first_to;

/**
 * Indicates if the text is selected or not
*/
  l_int          sel_ok;

/**
 * If true, cursor is visible, othewise is hiden
*/
  l_bool         cursor_visible;

  void          (*ins_text)( p_textline o, l_int pos, l_text text );
  void          (*ins_char)( p_textline o, l_int pos, l_char chr );
  void          (*del_text)( p_textline o, l_int pos, l_int size );
  void          (*del_char)( p_textline o, l_int pos );
  void          (*draw_cursor_ex)( p_textline o, l_int line, l_int pos, l_int active );
  void          (*draw_cursor)( p_textline o, l_bool sh );
  void          (*show_cursor)( p_textline o, l_bool show );
  void          (*draw_text)( p_textline o );
  void          (*redraw_text) ( p_textline o, l_int newpos, l_int keycode );
  void          (*sel_all) ( p_textline o, l_int selfrom, l_int selto );
  l_int         (*charsin) ( p_textline o, l_int from, l_int plus );
  l_int         (*charsin_size) ( p_textline o, l_int size, l_int from, l_int plus );
  l_int         (*get_pos_from_xy) ( p_textline o, t_point p );
  l_bool        (*set_text) ( p_textline o, l_text text );
  l_bool        (*is_in_select_rect) ( p_textline o, t_point where );
  t_rect        (*get_select_rect) ( p_textline o );

} t_textline;

extern p_textline def_textline;

#define TEXTLINE(o) ((p_textline)(o))

#define DEF_TEXTLINE_FUNC(y) DEF_FUNC(def_button,y)
#define DEF_TEXTLINE_VIEW_FUNC(y) DEF_FUNC(VIEW(def_textline),y)
#define DEF_TEXTLINE_VIEW_OBJECT_FUNC(y) DEF_FUNC(OBJECT(def_textline),y)

/* Clears textline text-selection */
#define sel_clear(o)    (o)->sel_from = (o)->sel_to = (o)->sel_first_from = (o)->sel_first_to = (o)->sel_ok = 0
#define sel_dd_clear(o) (o)->sel_safe.x = (o)->sel_safe.y = 0
/* Returns true if text is rewrite able, otherwise it returns false */
#define is_wable(o)  (!((o)->flags & TF_REWRITEUNABLE))
/* Returns true if text is selected */
#define is_sel(o)    ((o)->sel_from != (o)->sel_to)
#define is_sel_dd(o) ((o)->sel_safe.x != (o)->sel_safe.y)


/**
 * Creates the textline view, that enables you to write some data inside
 * one line. @limit is the maximum number of the characters that can be
 * inserted inside the textline.
 * @flags is the combination of the following flags that select the
 * properties of the text line.
*/
extern p_textline  (*textline_init) ( p_textline o, t_rect r, l_int limit, l_int flags );
p_textline  _textline_init ( p_textline o, t_rect r, l_int limit, l_int flags );


/**
 * Creates the textline view, that enables you to write some data inside
 * one line. @limit is the maximum number of the characters that can be
 * inserted inside the textline.
 * @flags is the combination of the following flags that select the
 * properties of the text line.
 *
 * This function is the same as above, but you couldn't use the memory
 * allocation. It's used manly from the high-level API. If you need to
 * make object that is inherited from the textline, please use the function
 * "textline_init"
*/
p_textline  textline_new ( t_rect r, l_int limit, l_int flags );


#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xtxtline.h */
