/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XGRAPHIC.H                                    */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of QubeOS                                  */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XGRAPHIC_H_INCLUDED__
#define __XGRAPHIC_H_INCLUDED__

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#include"basics.h"
#include"drvbas.h"


#define   CONV_BMP(x)     ((p_bitmap)(x->block))
#define   CONV_MTA(x)     ((t_meta*)(x->block))
#define   CONV_MTAFILE(x) (((t_meta*)(x->block))->file)

#define   FR_HOR     0x01
#define   FR_VER     0x02

/*
   Moznosti GRAPHICS
*/
#define GR_BITMAP  0x00001 /* kresli sa v pamati */
#define GR_SCREEN  0x00002 /* kresli sa priamo na obrazovku */
#define GR_META    0x00003 /* kresli sa do meta suboru */


/* Meta block options */

#define META_BLOCKLIMIT 0xFFFFFF  /* maximalna dlzka meta suboru */


/* t_meta - interna bitmapa */

typedef struct t_meta *p_meta;
typedef struct t_meta {

    l_rect     width;
    l_rect     height;
    l_int      depth;
    l_color    mask_color;
    l_rect     cax;
    l_rect     cay;
    l_rect     cbx;
    l_rect     cby;
    l_int      flags;
    p_file     file;

    l_char     reserved[32];

} t_meta;

typedef struct t_filler *p_filler;
typedef struct t_filler {

   l_int     mode;
   p_bitmap  pattern;
   l_int     x_anchor;
   l_int     y_anchor;

} t_filler;



typedef struct t_graphics *p_graphics;
typedef struct t_graphics {

   l_ptr     block;
   l_int     type;
   t_filler  brush;

   l_char    reserved[32];


/**2D**/
/*
   Each DC has an associated clipping rectangle, which is the area of
   the image that it is ok to draw on. Nothing will be drawn to positions 
   outside this space. Pass the two opposite corners of the clipping 
   rectangle: these are inclusive, eg. g->set_clip(g, 16, 16, 32, 32) will
   allow drawing to (16, 16) and (32, 32), but not to (15, 15) and (33, 33). 
   If x1, y1, x2, and y2 are all zero, clipping will be turned off, which 
   may slightly speed up some drawing operations (usually a negligible 
   difference, although every little helps) but will result in your program 
   dying a horrible death if you try to draw beyond the edges of the bitmap.
*/
   void     (*set_clip)(p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2);

/*
   Sets the graphics drawing mode. This only affects the geometric routines 
   like putpixel, lines, rectangles, circles, polygons, floodfill, etc, not 
   the text output, blitting, or sprite drawing functions. The mode should 
   be one of the values:

      DRAW_MODE_SOLID               - the default, solid color drawing
      DRAW_MODE_XOR                 - exclusive-or drawing
      DRAW_MODE_COPY_PATTERN        - multicolored pattern fill
      DRAW_MODE_SOLID_PATTERN       - single color pattern fill
      DRAW_MODE_MASKED_PATTERN      - masked pattern fill
      DRAW_MODE_TRANS               - translucent color blending

   In DRAW_MODE_XOR, pixels are written to the bitmap with an exclusive-or 
   operation rather than a simple copy, so drawing the same shape twice will 
   erase it. Because it involves reading as well as writing the bitmap 
   memory, xor drawing is a lot slower than the normal replace mode.

   With the patterned modes, you provide a pattern bitmap which is tiled 
   across the surface of the shape. Allegro stores a pointer to this bitmap 
   rather than copying it, so you must not destroy the bitmap while it is 
   still selected as the pattern. The width and height of the pattern must 
   be powers of two, but they can be different, eg. a 64x16 pattern is fine, 
   but a 17x3 one is not. The pattern is tiled in a grid starting at point 
   (x_anchor, y_anchor). Normally you should just pass zero for these 
   values, which lets you draw several adjacent shapes and have the patterns 
   meet up exactly along the shared edges. Zero alignment may look peculiar 
   if you are moving a patterned shape around the screen, however, because 
   the shape will move but the pattern alignment will not, so in some 
   situations you may wish to alter the anchor position.

   When you select DRAW_MODE_COPY_PATTERN, pixels are simply copied from the 
   pattern bitmap onto the destination bitmap. This allows the use of 
   multicolored patterns, and means that the color you pass to the drawing 
   routine is ignored. This is the fastest of the patterned modes.

   In DRAW_MODE_SOLID_PATTERN, each pixel in the pattern bitmap is compared 
   with the mask color, which is zero in 256 color modes or bright pink for 
   truecolor data (maximum red and blue, zero green). If the pattern pixel 
   is solid, a pixel of the color you passed to the drawing routine is 
   written to the destination bitmap, otherwise a zero is written. The 
   pattern is thus treated as a monochrome bitmask, which lets you use the 
   same pattern to draw different shapes in different colors, but prevents 
   the use of multicolored patterns.

   DRAW_MODE_MASKED_PATTERN is almost the same as DRAW_MODE_SOLID_PATTERN, 
   but the masked pixels are skipped rather than being written as zeros, so 
   the background shows through the gaps.

   In DRAW_MODE_TRANS, the global color_map table or truecolor blender 
   functions are used to overlay pixels on top of the existing image. This 
   must only be used after you have set up the color mapping table (for 256 
   color modes) or blender map (for truecolor modes). Because it involves 
   reading as well as writing the bitmap memory, translucent drawing is very 
   slow if you draw directly to video RAM, so wherever possible you should 
   use a memory bitmap instead.
*/
   p_filler (*drawing_mode)(p_graphics g, l_int mode, t_bitmap *pattern, l_int x_anchor, l_int y_anchor);

/*

*/
   void     (*reset_drawing_mode)(p_graphics g, p_filler *f );

/*
   Reads a pixel from point x, y in the bitmap. Returns -1 if the point lies
   outside the bitmap.
*/
   l_color  (*getpixel)(p_graphics g, l_rect x, l_rect y);

/*
   Writes a pixel to the specified position in the bitmap, using the current 
   drawing mode and the bitmap's clipping rectangle.
*/
   void     (*putpixel)(p_graphics g, l_rect x, l_rect y, l_color color);

/*
   Draws a vertical line onto the bitmap, from point (x, y1) to (x, y2).
*/
   void     (*vline)(p_graphics g, l_rect x, l_rect y1, l_rect y2, l_color color);

/*
   Draws a horizontal line onto the bitmap, from point (x1, y) to (x2, y).
*/
   void     (*hline)(p_graphics g, l_rect x1, l_rect y, l_rect x2, l_color color);

/*
   Draws a vertical line onto the bitmap, from point (x, y1) to (x, y2).
*/
   void     (*vdotline)(p_graphics g, l_rect x, l_rect y1, l_rect y2, l_int npix, l_color color);

/*
   Draws a horizontal line onto the bitmap, from point (x1, y) to (x2, y).
*/
   void     (*hdotline)(p_graphics g, l_rect x1, l_rect y, l_rect x2, l_int npix, l_color color);

   void     (*dotrect)(p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_int npix, l_color color);

/*
   Calculates all the points along a line from point (x1, y1) to (x2, y2), 
   calling the supplied function for each one. This will be passed a copy of 
   the bmp parameter, the x and y position, and a copy of the d parameter, 
   so it is suitable for use with putpixel().
*/
   void     (*line)(p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_color color);

/*
   Draws a solid, filled rectangle with the two points as its opposite 
   corners.
*/
   void     (*rectfill)(p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_color color);

/*
   Draws a copy of the sprite bitmap onto the destination bitmap at the 
   specified position. This is almost the same as blit(sprite, bmp, 0, 0, x, 
   y, sprite->w, sprite->h), but it uses a masked drawing mode where 
   transparent pixels are skipped, so the background image will show through 
   the masked parts of the sprite. Transparent pixels are marked by a zero 
   in 256 color modes or bright pink for truecolor data (maximum red and 
   blue, zero green).

   If the GFX_HW_VRAM_BLIT_MASKED bit in the gfx_capabilities flag is set, 
   the current driver supports hardware accelerated sprite drawing when the 
   source image is a video memory bitmap or a sub-bitmap of the screen. This 
   is extremely fast, so when this flag is set it may be worth storing some 
   of your more frequently used sprites in an offscreen portion of the video 
   memory.

   Warning: if the hardware acceleration flag is not set, draw_sprite() will 
   not work correctly when used with a video memory source image, and the 
   input graphic must always be a memory bitmap!

   Although generally not supporting graphics of mixed color depths, as a 
   special case this function can be used to draw 256 color source images 
   onto truecolor destination bitmaps, so you can use palette effects on 
   specific sprites within a truecolor program.
*/
   void     (*draw_sprite)(p_graphics g,  t_bitmap *sprite, l_rect x, l_rect y);

/*
   These are like draw_sprite(), but they flip the image about the vertical,
   horizontal, or diagonal, axis. This produces exact mirror images, which 
   is not the same as rotating the sprite (and it is a lot faster than the 
   rotation routine). The sprite must be a memory bitmap.
*/
   void     (*draw_sprite_v_flip)(p_graphics g,  t_bitmap *sprite, l_rect x, l_rect y);
   void     (*draw_sprite_h_flip)(p_graphics g,  t_bitmap *sprite, l_rect x, l_rect y);
   void     (*draw_sprite_vh_flip)(p_graphics g, t_bitmap *sprite, l_rect x, l_rect y);

/*
   Uses the global color_map table or truecolor blender functions to overlay 
   the sprite on top of the existing image. This must only be used after you 
   have set up the color mapping table (for 256 color modes) or blender map 
   (for truecolor modes). Because it involves reading as well as writing the 
   bitmap memory, translucent drawing is very slow if you draw directly to 
   video RAM, so wherever possible you should use a memory bitmap instead.
*/
   void     (*draw_trans_sprite)(p_graphics g, t_bitmap *sprite, l_rect x, l_rect y);

/*
   Tints the sprite image to the specified color or light level, using the 
   global color_map table, and draws the resulting image to the destination 
   bitmap. This must only be used after you have set up the color mapping 
   table (for 256 color modes) or blender map (for truecolor modes).
*/
   void     (*draw_lit_sprite)(p_graphics g, t_bitmap *sprite, l_rect x, l_rect y, l_color color);

/*
   Draws a copy of the sprite bitmap onto the destination bitmap at the 
   specified position, drawing transparent pixels in the current text mode 
   (skipping them if the text mode is -1, otherwise drawing them in the text 
   background color), and setting all other pixels to the specified color. 
   Transparent pixels are marked by a zero in 256 color modes or bright pink 
   for truecolor data (maximum red and blue, zero green). The sprite must be 
   an 8 bit image, even if the destination is a truecolor bitmap.
*/
   void     (*draw_character)(p_graphics g, t_bitmap *sprite, l_rect x, l_rect y, l_color color);

/*
   Clears the bitmap to the specified color.
*/
   void     (*clear_to_color)(p_graphics g, l_color color);

/*
   Calculates all the points along a line from point (x1, y1) to (x2, y2), 
   calling the supplied function for each one. This will be passed a copy of 
   the bmp parameter, the x and y position, and a copy of the d parameter, 
   so it is suitable for use with putpixel().
*/
   void     (*do_line)(p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_rect d, void (*proc)(t_bitmap *, l_rect, l_rect, l_color));

/*
   Draws a filled triangle between the three points.
*/
   void     (*triangle)(p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_rect x3, l_rect y3, l_color color);

/*
   Draws a filled polygon with an arbitrary number of corners. Pass the 
   number of vertices and an array containing a series of x, y points (a 
   total of vertices*2 values).
*/
   void     (*polygon)(p_graphics g, l_rect vertices, l_rect *points, l_color color);

/*
   Draws an outline rectangle with the two points as its opposite corners.
*/
   void     (*rect)(p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_color color);

/*
   Calculates all the points in a circle around point (x, y) with radius r, 
   calling the supplied function for each one. This will be passed a copy of 
   the bmp parameter, the x and y position, and a copy of the d parameter, 
   so it is suitable for use with putpixel().
*/
   void     (*do_circle)(p_graphics g, l_rect x, l_rect y, l_rect radius, l_rect d, void (*proc)(t_bitmap *, l_rect, l_rect, l_color));

/*
   Draws a circle with the specified centre and radius.
*/
   void     (*circle)(p_graphics g, l_rect x, l_rect y, l_rect radius, l_color color);

/*
   Draws a filled circle with the specified centre and radius.
*/
   void     (*circlefill)(p_graphics g, l_rect x, l_rect y, l_rect radius, l_color color);

/*
   Calculates all the points in an ellipse around point (x, y) with radius 
   rx and ry, calling the supplied function for each one. This will be 
   passed a copy of the bmp parameter, the x and y position, and a copy of 
   the d parameter, so it is suitable for use with putpixel().
*/
   void     (*do_ellipse)(p_graphics g, l_rect x, l_rect y, l_rect rx, l_rect ry, l_rect d, void (*proc)(t_bitmap *, l_rect, l_rect, l_color));

/*
   Draws an ellipse with the specified centre and radius.
*/
   void     (*ellipse)(p_graphics g, l_rect x, l_rect y, l_rect rx, l_rect ry, l_color color);

/*
   Draws a filled ellipse with the specified centre and radius.
*/
   void     (*ellipsefill)(p_graphics g, l_rect x, l_rect y, l_rect rx, l_rect ry, l_color color);

/*
   Calculates all the points in a circular arc around point (x, y) with
   radius r, calling the supplied function for each one. This will be passed 
   a copy of the bmp parameter, the x and y position, and a copy of the d 
   parameter, so it is suitable for use with putpixel(). The arc will be 
   plotted in an anticlockwise direction starting from the angle a1 and 
   ending when it reaches a2. These values are specified in 16.16 fixed 
   point format, with 256 equal to a full circle, 64 a right angle, etc. 
   Zero is to the right of the centre point, and larger values rotate 
   anticlockwise from there.
*/
   void     (*do_arc)(p_graphics g, l_rect x, l_rect y, l_fixed ang1, l_fixed ang2, l_rect r, l_rect d, void (*proc)(t_bitmap *, l_rect, l_rect, l_color));

/*
   Draws a circular arc with centre x, y and radius r, in an anticlockwise
   direction starting from the angle a1 and ending when it reaches a2. These 
   values are specified in 16.16 fixed point format, with 256 equal to a 
   full circle, 64 a right angle, etc. Zero is to the right of the centre 
   point, and larger values rotate anticlockwise from there.
*/
   void     (*arc)(p_graphics g, l_rect x, l_rect y, l_fixed ang1, l_fixed ang2, l_rect r, l_color color);

/*
   Draws a bezier spline using the four control points specified in the
   points array.
*/
   void     (*spline)(p_graphics g, l_rect points[8], l_color color);

/*
   Floodfills an enclosed area, starting at point (x, y), with the specified
   color.
*/
   void     (*floodfill)(p_graphics g, l_rect x, l_rect y, l_color color);

/*
   Like blit(), but skips transparent pixels, which are marked by a zero in 
   256 color modes or bright pink for truecolor data (maximum red and blue, 
   zero green). The source and destination regions must not overlap.

   If the GFX_HW_VRAM_BLIT_MASKED bit in the gfx_capabilities flag is set, 
   the current driver supports hardware accelerated masked blits from one 
   part of the screen onto another. This is extremely fast, so when this 
   flag is set it may be worth storing some of your more frequently used 
   sprites in an offscreen portion of the video memory.

   Warning: if the hardware acceleration flag is not set, masked_blit() will 
   not work correctly when used with a video memory source image, and the 
   input graphic must always be a memory bitmap!
*/
   void     (*masked_blit)(p_graphics g, t_bitmap *from, l_rect source_x, l_rect source_y, l_rect dest_x, l_rect dest_y, l_rect width, l_rect height);

/*
   Copies a rectangular area of the source bitmap to the destination bitmap. 
   The source_x and source_y parameters are the top left corner of the area 
   to copy from the source bitmap, and dest_x and dest_y are the 
   corresponding position in the destination bitmap. This routine respects 
   the destination clipping rectangle, and it will also clip if you try to 
   blit from areas outside the source bitmap.

   You can blit between any parts of any two bitmaps, even if the two memory 
   areas overlap (ie. source and dest are the same, or one is sub-bitmap of 
   the other). You should be aware, however, that a lot of SVGA cards don't 
   provide separate read and write banks, which means that blitting from one 
   part of the screen to another requires the use of a temporary bitmap in 
   memory, and is therefore extremely slow. As a general rule you should 
   avoid blitting from the screen onto itself in SVGA modes.

   In mode-X, on the other hand, blitting from one part of the screen to 
   another can be significantly faster than blitting from memory onto the 
   screen, as long as the source and destination are correctly aligned with 
   each other. Copying between overlapping screen rectangles is slow, but if 
   the areas don't overlap, and if they have the same plane alignment (ie. 
   (source_x%4) == (dest_x%4)), the VGA latch registers can be used for a 
   very fast data transfer. To take advantage of this, in mode-X it is often 
   worth storing tile graphics in a hidden area of video memory (using a 
   large virtual screen), and blitting them from there onto the visible part 
   of the screen.

   If the GFX_HW_VRAM_BLIT bit in the gfx_capabilities flag is set, the 
   current driver supports hardware accelerated blits from one part of the 
   screen onto another. This is extremely fast, so when this flag is set it 
   may be worth storing some of your more frequently used graphics in an 
   offscreen portion of the video memory.

   Unlike most of the graphics routines, blit() allows the source and 
   destination bitmaps to be of different color depths, so it can be used to 
   convert images from one pixel format to another.
*/
   void     (*blit)(p_graphics g, t_bitmap *from, l_rect source_x, l_rect source_y, l_rect dest_x, l_rect dest_y, l_rect width, l_rect height);

/*
   Like blit(), except it can scale images so the source and destination 
   rectangles don't need to be the same size. This routine doesn't do as 
   much safety checking as the regular blit: in particular you must take 
   care not to copy from areas outside the source bitmap, and you cannot 
   blit between overlapping regions, ie. you must use different bitmaps for 
   the source and the destination. Also, the source must be a memory bitmap 
   or sub-bitmap, not the hardware screen.
*/
   void     (*stretch_blit)(p_graphics g, t_bitmap *from, l_rect s_x, l_rect s_y, l_rect s_w, l_rect s_h, l_rect d_x, l_rect d_y, l_rect d_w, l_rect d_h);


/****/

/*
   Draws the sprite image onto the bitmap at the specified position, 
   stretching it to the specified width and height. The difference between 
   stretch_sprite() and stretch_blit() is that stretch_sprite() masks out 
   transparent pixels, which are marked by a zero in 256 color modes or 
   bright pink for truecolor data (maximum red and blue, zero green).
*/
   void     (*stretch_sprite)(p_graphics g, t_bitmap *sprite, l_rect x, l_rect y, l_rect w, l_rect h);

/*
   Draws the sprite image onto the bitmap at the specified position, 
   rotating it by the specified angle. The angle is a fixed point 16.16 
   number in the same format used by the fixed point trig routines, with 256 
   equal to a full circle, 64 a right angle, etc. The sprite must be a 
   memory bitmap.
*/
   void     (*rotate_sprite)(p_graphics g, t_bitmap *sprite, l_rect x, l_rect y, l_fixed angle);

/*
   Like rotate_sprite(), but stretches or shrinks the image at the same time
   as rotating it.
*/
   void     (*rotate_scaled_sprite)(p_graphics g, t_bitmap *sprite, l_rect x, l_rect y, l_fixed angle, l_fixed scale);

/*
   Tints the sprite to the specified color or light level, interpolating the
   four corner colors across the surface of the image. This must only be 
   used after you have set up the color mapping table (for 256 color modes) 
   or blender map (for truecolor modes).
*/
   void     (*draw_gouraud_sprite)(p_graphics g, t_bitmap *sprite, l_rect x, l_rect y, l_rect c1, l_rect c2, l_rect c3, l_rect c4);

/*
   Clears the bitmap to color 0.
*/
   void     (*clear)(p_graphics g);

/*
   Fade and fill rectangle from color one to color two. Direction can
   be one of the following bits :

   #define FR_HOR  - fade rectangle from top to bottom

   #define FR_VER  - fade rectangle from left to right

*/
   void     (*fade_rect) ( p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_color one, l_color two, l_char direct );

/*
   Vypln stvorec ciarami, kde ciara je zlozena z dvoch
   horizontalnych ciar, kde prva ma farbu one a druha farbu two.
*/
   void     (*lined_rect) ( p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_color one, l_color two );

/*
   Zosvetli priestor g v cervenej o dr, v zelenej o dg a modrej o db.
*/
   void     (*light_image) ( p_graphics g, p_bitmap sprite, l_rect x1, l_rect y1, l_byte dr, l_byte dg, l_byte db );


/**
 * Makes alphas image with a color @rr,@gg,@bb and blends this image with
 * the alpha index @alpha as the range from 0..100
*/
   void     (*alpha_image) ( p_graphics g, p_bitmap sprite, l_rect dx, l_rect dy, l_rect sx1, l_rect sy1, l_rect sx2, l_rect sy2, l_byte dr, l_byte dg, l_byte db, l_int alpha );

/*
   Kresli tlacitko, ktoreho lava a horna ciara bude definovana farbou lt
   a prava dolna ciara farbou rb.
*/
   void     (*button) ( p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_color lt, l_color rb );


/*
   Kresli stvorec vyplneny bitmapou v "sprite"
*/
   void     (*sprite_rectfill) ( p_graphics g, t_bitmap *sprite, l_rect x1, l_rect y1, l_rect x2, l_rect y2 );


/*
   Kresli okruhle tlacitko, ktoreho lava a horna ciara bude definovana farbou lt
   a prava dolna ciara farbou rb.
*/
   void     (*arc_button) ( p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_rect radius, l_color lt, l_color rb );

/**3D**/
/*
   Draw 3d polygons onto the specified bitmap, using the specified rendering 
   mode. Unlike the regular polygon() function, these routines don't support 
   concave or self-intersecting shapes, and they can't draw onto mode-X 
   screen bitmaps (if you want to write 3d code in mode-X, draw onto a 
   memory bitmap and then blit to the screen). The width and height of the 
   texture bitmap must be powers of two, but can be different, eg. a 64x16 
   texture is fine, but a 17x3 one is not. The vertex count parameter (vc) 
   should be followed by an array containing the appropriate number of 
   pointers to vertex structures: polygon3d() uses the fixed point V3D 
   structure, while polygon3d_f() uses the floating point V3D_f structure. 
   These are defined as:

   typedef struct V3D
   {
      fixed x, y, z;       - position
      fixed u, v;          - texture map coordinates
      int c;               - color
   } V3D;

   typedef struct V3D_f
   {
      float x, y, z;       - position
      float u, v;          - texture map coordinates
      int c;               - color
   } V3D_f;

   How the vertex data is used depends on the rendering mode:

   The x and y values specify the position of the vertex in 2d screen 
   coordinates.

   The z value is only required when doing perspective correct texture 
   mapping, and specifies the depth of the point in 3d world coordinates.

   The u and v coordinates are only required when doing texture mapping, and 
   specify the position of the point in the texture bitmap, for example 0, 0 
   maps the vertex onto the top left corner of the texture, and if the 
   texture is sized 32x32, setting u=32 and v=16 maps the vertex to the 
   point half way down the right edge of the texture. The u/v coordinates 
   wrap at the edge of the texture, so with a 32x32 texture, u=v=32 is the 
   same as u=v=0. This can be used to tile textures several times across a 
   polygon.

   The c value specifies the vertex color, and is interpreted differently by 
   various rendering modes.

   The type parameter specifies the polygon rendering mode, and can be any 
   of the values:

   POLYTYPE_FLAT:
      A simple flat shaded polygon, taking the color from the c value of the 
      first vertex. This polygon type is affected by the drawing_mode() 
      function, so it can be used to render XOR or translucent polygons.

   POLYTYPE_GCOL:
      A single-color gouraud shaded polygon. The colors for each vertex are 
      taken from the c value, and interpolated across the polygon. This is 
      very fast, but will only work in 256 color modes if your palette 
      contains a smooth gradient between the colors. In truecolor modes it 
      interprets the color as a packed, display-format value as produced by 
      the makecol() function.

   POLYTYPE_GRGB:
      A gouraud shaded polygon which interpolates RGB triplets rather than a 
      single color. In 256 color modes this uses the global rgb_map table to 
      convert the result to an 8 bit paletted color, so it must only be used 
      after you have set up the RGB mapping table! The colors for each 
      vertex are taken from the c value, which is interpreted as a 24 bit 
      RGB triplet (0xFF0000 is red, 0x00FF00 is green, and 0x0000FF is 
      blue).

   POLYTYPE_ATEX:
      An affine texture mapped polygon. This stretches the texture across 
      the polygon with a simple 2d linear interpolation, which is fast but 
      not mathematically correct. It can look ok if the polygon is fairly 
      small or flat-on to the camera, but because it doesn't deal with 
      perspective foreshortening, it can produce strange warping artifacts. 
      To see what I mean, run test.exe and see what happens to the 
      polygon3d() test when you zoom in very close to the cube.

   POLYTYPE_PTEX:
      A perspective-correct texture mapped polygon. This uses the z value 
      from the vertex structure as well as the u/v coordinates, so textures 
      are displayed correctly regardless of the angle they are viewed from. 
      Because it involves division calculations in the inner texture mapping 
      loop, this mode is a lot slower than POLYTYPE_ATEX, and it uses 
      floating point so it will be very slow on anything less than a Pentium 
      (even with an FPU, a 486 can't overlap floating point division with 
      other integer operations like the Pentium can).

   POLYTYPE_ATEX_MASK:
   POLYTYPE_PTEX_MASK:
      Like POLYTYPE_ATEX and POLYTYPE_PTEX, but zero texture map pixels are 
      skipped, allowing parts of the texture map to be transparent.

   POLYTYPE_ATEX_LIT:
   POLYTYPE_PTEX_LIT:
      Like POLYTYPE_ATEX and POLYTYPE_PTEX, but the global color_map table 
      (for 256 color modes) or blender function (for non-MMX truecolor 
      modes) is used to blend the texture with a light level taken from the 
      c value in the vertex structure. This must only be used after you have 
      set up the color mapping table or blender functions!

   POLYTYPE_ATEX_MASK_LIT:
   POLYTYPE_PTEX_MASK_LIT:
      Like POLYTYPE_ATEX_LIT and POLYTYPE_PTEX_LIT, but zero texture map 
      pixels are skipped, allowing parts of the texture map to be 
      transparent.

   If the cpu_mmx global variable is set, the GRGB and truecolor *LIT 
   routines will be optimised using MMX instructions. If the cpu_3dnow 
   global variable is set, the truecolor PTEX*LIT routines will take 
   advantage of the 3DNow! CPU extensions. For this reason, it is a good 
   idea to call check_cpu() before using the polygon functions.

   Using MMX for *LIT routines has a side effect: normally (without MMX), 
   these routines use the blender functions used also for other lighting 
   functions, set with set_trans_blender() or set_blender_mode(). The MMX 
   versions only use the RGB value passed to set_trans_blender() and do the 
   linear interpolation themselves. Therefore a new set of blender functions 
   passed to set_blender_mode() is ignored.
*/
   void     (*polygon3d)(p_graphics g, l_int type, t_bitmap *texture, l_int vc, p_v3d vtx[]);
   void     (*polygon3d_f)(p_graphics g, l_int type, t_bitmap *texture, l_int vc, p_v3df vtx[]);

/*
   Draw 3d triangles, using either fixed or floating point vertex 
   structures. These are equivalent to calling
      polygon3d(bmp, type, tex, 3, v1, v2, v3);
   or
      polygon3d_f(bmp, type, tex, 3, v1, v2, v3);
*/
   void     (*triangle3d)(p_graphics g, l_int type, t_bitmap *texture, p_v3d v1, p_v3d v2, p_v3d v3);
   void     (*triangle3d_f)(p_graphics g, l_int type, t_bitmap *texture, t_v3df *v1, t_v3df *v2, t_v3df *v3);

/*
   Draw 3d quads, using either fixed or floating point vertex structures.
   These are equivalent to calling
      polygon3d(bmp, type, tex, 4, v1, v2, v3, v4);
   or
      polygon3d_f(bmp, type, tex, 4, v1, v2, v3, v4);
*/
   void     (*quad3d)(p_graphics g, l_int type, t_bitmap *texture, t_v3d *v1, t_v3d *v2, t_v3d *v3, t_v3d *v4);
   void     (*quad3d_f)(p_graphics g, l_int type, t_bitmap *texture, t_v3df *v1, t_v3df *v2, t_v3df *v3, t_v3df *v4);

/**INFO**/
/*
   Zisti farebne rozatie g. Vrati hodnotu 8 pre 256 farieb,
   15 pre 32768 farieb, 16 pre 65636 farieb a 24 pre 16M farieb.
*/
   l_int    (*get_depth) ( p_graphics g );

/*
   Vracia farbu, ktora sa v tomto g nezobrazuje.
*/
   l_color  (*get_mask_color) ( p_graphics g );

/*
   Zisti sirku g.
*/
   l_rect   (*get_width) ( p_graphics g );

/*
   Zisti vysku g.
*/
   l_rect   (*get_height) ( p_graphics g );

/*
   Zisti platny stvorec g, v ktorom sa grafika zobrazuje.
*/
   l_bool   (*get_clip) ( p_graphics g, l_rect *x1, l_rect *y1, l_rect *x2, l_rect *y2 );

/*
   Returns linear pointer to graphics. Use only in case, your graphics
   uses direct access.
*/
   l_ptr    (*get_ptr) ( p_graphics g );

/**TEXT**/

/*
   Vypis znak chr definovany fontom f na poziciu x,y, kde pozadie
   textu je vyplnene farbou bg a popredie farbou fg. Ak nechceme popredie,
   alebo pozadie zobrazovat pouzijeme farbu CO_NOCOLOR.
*/
  void      (*text_draw_char) ( p_graphics g, p_font f, l_char chr, l_rect x, l_rect y, l_color fg, l_color bg );

/*
   Vypis nulov zakonceny text s definovany fontom f na poziciu x,y,
   kde pozadie textu je vyplnene farbou bg a popredie farbou fg.
   Ak nechceme popredie, alebo pozadie zobrazovat pouzijeme farbu CO_NOCOLOR.
   Text sa automaticky zarovna podla kombinacie bitov v align.

   #define  TX_ALIGN_LEFT     - zarovnaj text podla x suradnice dolava.
                              ( sdandardne )
   #define  TX_ALIGN_TOP      - zarovnaj text podla y suradnice hore.
                              ( sdandardne )
   #define  TX_ALIGN_RIGHT    - zarovnaj text podla x suradnice doprava .
   #define  TX_ALIGN_BOTTOM   - zarovnaj text podla y suradnice dole.
   #define  TX_ALIGN_CENTERX  - zarovnaj text podla x suradnice do stredu.
   #define  TX_ALIGN_CENTERY  - zarovnaj text podla y suradnice do stredu.
   #define  TX_ALIGN_CENTER   - zarovnaj text podla x,y suradnice do stredu.
   #define  TX_ALIGN_DEFAULT  - zarovnaj text podla x,y suradnice dolava,hore.
   #define  TX_UNDERLINE      - text bude pogiarknuty.
   #define  TX_STRIKELINE     - text bude preskrknuty
   #define  TX_NOSMOOTHTEXT

*/
   void     (*text_draw) ( p_graphics g, p_font f, l_text s, l_int len, l_rect x, l_rect y, l_int align, l_color fg, l_color bg );


/*
   Vypis nulov zakonceny text s definovany fontom f do stvorca x1,y1,x2,y2
   kde pozadie textu je vyplnene farbou bg a popredie farbou fg.
   Ak nechceme popredie, alebo pozadie zobrazovat pouzijeme farbu CO_NOCOLOR.
   Text sa automaticky zarovna podla kombinacie bitov v align.

   #define  TX_ALIGN_LEFT     - zarovnaj text podla x suradnice dolava.
                              ( sdandardne )
   #define  TX_ALIGN_TOP      - zarovnaj text podla y suradnice hore.
                              ( sdandardne )
   #define  TX_ALIGN_RIGHT    - zarovnaj text podla x suradnice doprava .
   #define  TX_ALIGN_BOTTOM   - zarovnaj text podla y suradnice dole.
   #define  TX_ALIGN_CENTERX  - zarovnaj text podla x suradnice do stredu.
   #define  TX_ALIGN_CENTERY  - zarovnaj text podla y suradnice do stredu.
   #define  TX_ALIGN_CENTER   - zarovnaj text podla x,y suradnice do stredu.
   #define  TX_ALIGN_DEFAULT  - zarovnaj text podla x,y suradnice dolava,hore.
   #define  TX_UNDERLINE      - text bude podciarknuty.
   #define  TX_STRIKELINE     - text bude preskrknuty
   #define  TX_NOSMOOTHTEXT

*/
   void     (*text_draw_rect) ( p_graphics g, p_font f, l_text s, l_int len, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_int align, l_color fg, l_color bg, l_bool clip );

   void     (*text_draw_double) ( p_graphics g, p_font f, l_text txt, l_int size, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_int align, l_color fg1, l_color fg2, l_bool clip );
   void     (*text_draw_selected) ( p_graphics g, p_font f, l_text txt, l_int size, l_int sfrom, l_int sto, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_int align, l_color fg1, l_color bg1, l_color fg2, l_color bg2, l_bool clip );


/** t_graphics BLITTING **/

   void     (*masked_blit_graphics)(p_graphics g, p_graphics from, l_rect source_x, l_rect source_y, l_rect dest_x, l_rect dest_y, l_rect width, l_rect height);

   void     (*blit_graphics)(p_graphics g, p_graphics from, l_rect source_x, l_rect source_y, l_rect dest_x, l_rect dest_y, l_rect width, l_rect height);

   void     (*stretch_blit_graphics)(p_graphics g, p_graphics from, l_rect s_x, l_rect s_y, l_rect s_w, l_rect s_h, l_rect d_x, l_rect d_y, l_rect d_w, l_rect d_h);


/* don't call these ( hidden ) */
   void     (*masked_blit_graphics_me) (p_graphics g, p_graphics to, l_rect source_x, l_rect source_y, l_rect dest_x, l_rect dest_y, l_rect width, l_rect height);
   void     (*blit_graphics_me) (p_graphics g, p_graphics to, l_rect source_x, l_rect source_y, l_rect dest_x, l_rect dest_y, l_rect width, l_rect height);
   void     (*stretch_blit_graphics_me) ( p_graphics g, p_graphics to, l_rect s_x, l_rect s_y, l_rect s_w, l_rect s_h, l_rect d_x, l_rect d_y, l_rect d_w, l_rect d_h);

/* begin paint + end of paint **/
   void     (*begin_paint) ( p_graphics g );
   void     (*end_of_paint) ( p_graphics g );

/* creating */
/*
  Creates compatible graphics that has the same depth as @g
*/
   p_graphics (*create_compatible_ex) ( p_graphics g, l_int type, l_rect w, l_rect h );

/*
  Creates compatible graphics that has the same depth as @g, and is in the
  same type.
*/
   p_graphics (*create_compatible) ( p_graphics g, l_rect w, l_rect h );

/*
   Kresli zaobleny stvorec
*/
   void     (*arc_rectfill) ( p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_rect radius, l_color color );

/*
   Kresli zaobleny stvorec, kde vonkajsia cast je vyplnena farbou c2
*/
   void     (*arc_rectfill_ex) ( p_graphics g, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_rect radius, l_color color, l_color c2 );

} t_graphics;

#define GRAPHICS(p) ((p_graphics)(o))

extern p_graphics def_graphics_bmpscr;
extern p_graphics def_graphics_meta;

#define DEF_GRAPHICS_BMPSCR_FUNC(y) DEF_GRAPHICS_BMPSCR_FUNC(def_graphics_bmpscr,y)
#define DEF_GRAPHICS_META_FUNC(y)   DEF_GRAPHICS_META_FUNC(def_graphics_meta,y)


#define   gr_create_compatible_ex(g,t,w,h)\
          g->create_compatible_ex(g,t,w,h)

#define   gr_create_compatible(g,w,h)\
          g->create_compatible(g,w,h)

#define   gr_create_compatible_size(g,t)\
          gr_create_compatible_ex(g,t,gr_get_width(g),gr_get_height(g))

#define   gr_set_clip(g,x1,y1,x2,y2)\
          g->set_clip(g,x1,y1,x2,y2)

#define   gr_reset_clip(g) \
          g->set_clip(g, 0, 0, g->get_width(g)-1, g->get_height(g)-1)

#define   gr_drawing_mode(g,mode,pattern,x_anchor,y_anchor)\
          g->drawing_mode(g,mode,pattern,x_anchor,y_anchor)

#define   gr_reset_drawing_mode(g,f)\
          g->reset_drawing_mode(g,f)

#define   gr_getpixel(g,x,y)\
          g->getpixel(g,x,y)

#define   gr_putpixel(g,x,y,color)\
          g->putpixel(g,x,y,color)

#define   gr_vline(g,x,y1,y2,color)\
          g->vline(g,x,y1,y2,color)

#define   gr_hline(g,x1,y,x2,color)\
          g->hline(g,x1,y,x2,color)

#define   gr_vdotline(g,x,y1,y2,npix,color)\
          g->vdotline(g,x,y1,y2,npix,color)

#define   gr_hdotline(g,x1,y,x2,npix,color)\
          g->hdotline(g,x1,y,x2,npix,color)

#define   gr_line(g,x1,y1,x2,y2,color)\
          g->line(g,x1,y1,x2,y2,color)

#define   gr_rectfill(g,x1,y1,x2,y2,color)\
          g->rectfill(g,x1,y1,x2,y2,color)

#define   gr_draw_sprite(g,sprite,x,y)\
          g->draw_sprite(g,sprite,x,y)

#define   gr_draw_sprite_v_flip(g,sprite,x,y)\
          g->draw_sprite_v_flip(g,sprite,x,y)

#define   gr_draw_sprite_h_flip(g,sprite,x,y)\
          g->draw_sprite_h_flip(g,sprite,x,y)

#define   gr_draw_sprite_vh_flip(g,sprite,x,y)\
          g->draw_sprite_vh_flip(g,sprite,x,y)

#define   gr_draw_trans_sprite(g,sprite,x,y)\
          g->draw_trans_sprite(g,sprite,x,y)

#define   gr_draw_lit_sprite(g,sprite,x,y,color)\
          g->draw_lit_sprite(g,sprite,x,y,color)

#define   gr_draw_character(g,sprite,x,y,color)\
          g->draw_character(g,sprite,x,y,color)

#define   gr_clear_to_color(g,color)\
          g->clear_to_color(g,color)

#define   gr_do_line(g,x1,y1,x2,y2,d,proc)\
          g->do_line(g,x1,y1,x2,y2,d,proc)

#define   gr_triangle(g,x1,y1,x2,y2,x3,y3,color)\
          g->triangle(g,x1,y1,x2,y2,x3,y3,color)

#define   gr_polygon(g,vertices,points,color)\
          g->polygon(g,vertices,points,color)

#define   gr_rect(g,x1,y1,x2,y2,color)\
          g->rect(g,x1,y1,x2,y2,color)

#define   gr_dotrect(g,x1,y1,x2,y2,npix,color)\
          g->dotrect(g,x1,y1,x2,y2,npix,color)

#define   gr_do_circle(g,x,y,radius,d,proc)\
          g->do_circle(g,x,y,radius,d,proc)

#define   gr_circle(g,x,y,radius,color)\
          g->circle(g,x,y,radius,color)

#define   gr_circlefill(g,x,y,radius,color)\
          g->circlefill(g,x,y,radius,color)

#define   gr_do_ellipse(g,x,y,rx,ry,d,proc)\
          g->do_ellipse(g,x,y,rx,ry,d,proc)

#define   gr_ellipse(g,x,y,rx,ry,color)\
          g->ellipse(g,x,y,rx,ry,color)

#define   gr_ellipsefill(g,x,y,rx,ry,color)\
          g->ellipsefill(g,x,y,rx,ry,color)

#define   gr_do_arc(g,x,y,ang1,ang2,r,d,proc)\
          g->do_arc(g,x,y,ang1,ang2,r,d,proc)

#define   gr_arc(g,x,y,ang1,ang2,r,color)\
          g->arc(g,x,y,ang1,ang2,r,color)

#define   gr_spline(g,points,color)\
          g->spline(g,points,color)

#define   gr_floodfill(g,x,y,color)\
          g->floodfill(g,x,y,color)

#define   gr_masked_blit(g,from,source_x,source_y,dest_x,dest_y,width,height)\
          g->masked_blit(g,from,source_x,source_y,dest_x,dest_y,width,height)

#define   gr_blit(g,from,source_x,source_y,dest_x,dest_y,width,height)\
          g->blit(g,from,source_x,source_y,dest_x,dest_y,width,height)

#define   gr_stretch_blit(g,from,s_x,s_y,s_w,s_h,d_x,d_y,d_w,d_h)\
          g->stretch_blit(g,from,s_x,s_y,s_w,s_h,d_x,d_y,d_w,d_h)

#define   gr_stretch_sprite(g,sprite,x,y,w,h)\
          g->stretch_sprite(g,sprite,x,y,w,h)

#define   gr_rotate_sprite(g,sprite,x,y,angle)\
          g->rotate_sprite(g,sprite,x,y,angle)

#define   gr_rotate_scaled_sprite(g,sprite,x,y,angle,scale)\
          g->rotate_scaled_sprite(g,sprite,x,y,angle,scale)

#define   gr_draw_gouraud_sprite(g,sprite,x,y,c1,c2,c3,c4)\
          g->draw_gouraud_sprite(g,sprite,x,y,c1,c2,c3,c4)

#define   gr_clear(g)\
          g->clear(g)

#define   gr_fade_rect(g,x1,y1,x2,y2,one,two,direct)\
          g->fade_rect(g,x1,y1,x2,y2,one,two,direct)

#define   gr_lined_rect(g,x1,y1,x2,y2,one,two)\
          g->lined_rect(g,x1,y1,x2,y2,one,two)

#define   gr_light_image(g,s,x1,y1,dr,dg,db) \
          g->light_image(g,s,x1,y1,dr,dg,db)

#define   gr_alpha_image(g,s,dx,dy,sx1,sy1,sx2,sy2,dr,dg,db,alpha)\
          g->alpha_image(g,s,dx,dy,sx1,sy1,sx2,sy2,dr,dg,db,alpha)

#define   gr_button(g,x1,y1,x2,y2,lt,rb) \
          g->button(g,x1,y1,x2,y2,lt,rb)

#define   gr_sprite_rectfill(g,s,x1,y1,x2,y2) \
          g->sprite_rectfill(g,s,x1,y1,x2,y2)

#define   gr_arc_button(g,x1,y1,x2,y2,r,lt,rb) \
          g->arc_button(g,x1,y1,x2,y2,r,lt,rb)

#define   gr_arc_rectfill(g,x1,y1,x2,y2,r,c) \
          g->arc_rectfill(g,x1,y1,x2,y2,r,c)

#define   gr_arc_rectfill_ex(g,x1,y1,x2,y2,r,c,c2) \
          g->arc_rectfill_ex(g,x1,y1,x2,y2,r,c,c2)

#define   gr_polygon3d(g,type,texture,vc,vtx) \
          g->polygon3d(g,type,texture,vc,vtx)

#define   gr_polygon3d_f(g,type,texture,vc,vtx) \
          g->polygon3d_f(g,type,texture,vc,vtx)

#define   gr_triangle3d(g,type,texture,v1,v2,v3) \
          g->triangle3d(g,type,texture,v1,v2,v3)

#define   gr_triangle3d_f(g,type,texture,v1,v2,v3) \
          g->triangle3d_f(g,type,texture,v1,v2,v3)

#define   gr_quad3d(g,type,texture,v1,v2,v3,v4) \
          g->quad3d(g,type,texture,v1,v2,v3,v4)

#define   gr_quad3d_f(g,type,texture,v1,v2,v3,v4) \
          g->quad3d_f(g,type,texture,v1,v2,v3,v4)

#define   gr_get_depth(g) \
          g->get_depth(g)

#define   gr_get_mask_color(g) \
          g->get_mask_color(g)

#define   gr_get_width(g) \
          g->get_width(g)

#define   gr_get_height(g) \
          g->get_height(g)

#define   gr_get_clip(g,x1,y1,x2,y2) \
          g->get_clip(g,x1,y1,x2,y2)

#define   gr_get_ptr(g) \
          g->get_ptr(g)

#define   gr_text_draw_char(g,f,chr,x,y,fg,bg)\
          g->text_draw_char(g,f,chr,x,y,fg,bg)

#define   gr_text_draw(g,f,s,len,x,y,align,fg,bg) \
          g->text_draw(g,f,s,len,x,y,align,fg,bg)

#define   gr_text_draw_rect(g,f,s,len,x1,y1,x2,y2,align,fg,bg,clip) \
          g->text_draw_rect(g,f,s,len,x1,y1,x2,y2,align,fg,bg,clip)

#define   gr_text_draw_double(g,f,txt,size,x1,y1,x2,y2,align,fg1,fg2,clip) \
          g->text_draw_double(g,f,txt,size,x1,y1,x2,y2,align,fg1,fg2,clip)

#define   gr_text_draw_selected(g,f,txt,size,sfrom,sto,x1,y1,x2,y2,align,fg1,bg1,fg2,bg2,clip) \
          g->text_draw_selected(g,f,txt,size,sfrom,sto,x1,y1,x2,y2,align,fg1,bg1,fg2,bg2,clip)

#define   gr_masked_blit_graphics(g,from,source_x,source_y,dest_x,dest_y,width,height) \
          do { if (g) g->masked_blit_graphics(g,from,source_x,source_y,dest_x,dest_y,width,height);}while(0)

#define   gr_blit_graphics(g,from,source_x,source_y,dest_x,dest_y,width,height) \
          do { if (g&&from) g->blit_graphics(g,from,source_x,source_y,dest_x,dest_y,width,height);}while(0)

#define   gr_stretch_blit_grpahics(g,from,s_x,s_y,s_w,s_h,d_x,d_y,d_w,d_h) \
          do { if (g&&from) g->stretch_blit_grpahics(g,from,s_x,s_y,s_w,s_h,d_x,d_y,d_w,d_h);}while(0)

#define   gr_begin_paint(g) \
          do { if (g) g->begin_paint(g); } while(0)

#define   gr_end_of_paint(g) \
          do { if (g) g->end_of_paint(g); } while(0)

/*
  Creates compatible graphics of type @t in depth the same to screen with
  width @w and height @h
*/
#define graphics_create(t,w,h) gr_create_compatible_ex(gr_screen, t, w, h)

/*
  Creates bitmap compatible graphics with screen depth
*/
#define graphics_create_memory(w,h)  graphics_create(GR_BITMAP, w, h)

/*
  Creates meta compatible graphics with screen depth
*/
#define graphics_create_meta(w,h) graphics_create(GR_META, w, h)

/*
  Initialize graphics by using the same bitmap @b. By this way you can
  own more graphics shared one bitmap.

  Returns new graphics.
*/
extern p_graphics  (*graphics_init_bitmap) ( p_bitmap b );

/*
  Installs graphics of type LOBYTE(type) and @width, @height, @depth.
  Avaialable types are :

  GR_SCREEN - can call only once. HIBYTE(type) contains information about
              screen driver.
  GR_BITMAP - install bitmap graphics. All graphics operation are drawn into
              memory, from where can blit to the screen by using
              gr_blit_graphics, gr_masked_blit_graphics,
              gr_stretch_blit_graphics
  GR_META   - install meta graphics. All graphics operation are drawn into
              the file/memory or another protocol. This is vector graphics.
*/
extern p_graphics  (*graphics_init) ( l_int type, l_rect width, l_rect height, l_int depth );

/*
  Destroy graphics and returns true on success.
*/
extern l_bool      (*graphics_destroy) ( p_graphics g );


/* internals */

extern void  (*def_standard_graphics_proc) ( p_graphics p );
extern void  (*def_meta_graphics_proc) ( p_graphics p );

#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xgraphic.h */

