/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XPROTOCL.H                                    */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of MatrixOS                                */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XPROTOCL_H_INCLUDED__
#define __XPROTOCL_H_INCLUDED__

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#include"basics.h"
#include"drvbas.h"

#define ERRPROTOCOL_FUNCTION_NOT_SUPPORTED   -1000

#define INFPROTOCOL_NEED_LINKTODIRECTORY      1
#define INFPROTOCOL_1ST_LEVEL_DIRECTORY       2

/*  */
#define  F_PROTOCOL_SPECASESENS        1

typedef  t_lowfilestamp  t_filestamp;
typedef  t_lowfindresult t_findresult;
typedef  l_ptr                  l_protocolfile;

typedef struct t_protocol *p_protocol;
typedef struct t_protocol {

   /*
      It's used for info that comes from function and can't be returned
      by the function because of other returning type. It's only a local
      function info, that usually contains error that was occured.
   */
   l_long   info;

   /*
      Flags of protocol : like case sensitivity & others...
      see F_SPECxxx in _depio.h
      localhost must contain the same values like in low_io_specification,
      so it copyies low_io_specification variable to this flags
   */
   l_long   flags;

   /*
     The size of transfer_buffer that's used for transfering bytes in this
     protocol
   */
   l_long   transfer_size;

   /*
     Prefix is the name of the protocol e.g. http, ftp, localhost, ...
   */
   l_text   prefix;

   /*
     Open file for writing, reading, that's defined by combinations
     in @mode param.

     F_READ  - Open an existing file for reading.
     F_WRITE - Create a new file (or truncate an existing file) and open it for
               writing.
     F_BINARY - Open binary file
     F_TEXT   - Open text file

     Returns pointer to opened file. In localhost it's l_lowfile,
     what's in C low-level FILE structure.
   */
   l_protocolfile      (*open)(p_protocol p, cl_text filename, l_int mode);

   /*
     This function reads @size characters from @file to @ptr.
     Must returns the @size read, or less if there was an error.
   */
   l_long              (*read)(p_protocol p, l_protocolfile file, l_ptr ptr, l_long size);

   /*
     This function writes @size characters from @file to @ptr.
     Must returns the @size read, or less if there was an error.
   */
   l_long              (*write)(p_protocol p, l_protocolfile file, l_ptr ptr, l_long size);

   /*
     Writes any buffered output to file opened for output.
   */
   l_int               (*flush)(p_protocol p, l_protocolfile file);

   /*
     Closes file that was previous opened by "open" function
   */
   l_int               (*close)(p_protocol p, l_protocolfile file);

   /*
     Returns length of the file that was opened by "open" function
   */
   l_long              (*length)(p_protocol p, l_protocolfile file);

   /*
     Tests if the position of the file is on the end of the file
   */
   l_int               (*eof)(p_protocol p, l_protocolfile file);

   /*
     Sets the position of the file, that was opened by "open" function
   */
   l_int               (*set_pos)(p_protocol p, l_protocolfile file, l_long where );

   /*
     Gets current position in the file, that was opened by "open" function
   */
   l_long              (*get_pos)(p_protocol p, l_protocolfile file );


   /*
     Function translates path from univerzal file system to path that can
     read protocol e.g. :

     l_text m = localhost->translate_path(localhost, "loclahost://dev/c/system");

     'c:\\system'
   */
   l_text     (*translate_path) ( p_protocol p, l_text path, l_int flag );
   l_text     (*content_type)(p_protocol o, cl_text file);

   /*
     Sets the file stamp of the file (path).
     t_filestamp has the following structure :

     l_time mtime - last modification time
     l_time atime - last access time
     l_time ctime - creation time

     You can convert from/to (l_time) that's in long format to readable one
     by using low_io_timer_xxx functions.
   */
   l_int      (*get_file_stamp)(p_protocol o, cl_text path, t_filestamp *t );
   l_int      (*set_file_stamp)(p_protocol o, cl_text path, t_filestamp t );

   /*
     If you want to list all files in the directory @path, you should
     use function findfirst only for first finding and then findnext for
     all findning in the same directory. Finding must be enclosed by
     function findclose, that free all memory used in findfirst/findnext
   */
   l_int      (*findfirst)(p_protocol p, cl_text path, t_findresult* res, l_int attrib );
   l_int      (*findnext)(p_protocol p, t_findresult* res );
   l_int      (*findclose)(p_protocol p, t_findresult* res);

   /*
     Returns pointer where data of the file are stored. Usually NULL for
     media, disks. Not NULL in the case of memory. See xmemory.h

     It's used in the case of the protocol can manage only
     one file. Made for compatibility of memory protocol with others.

     Please be carefull, because the pointer returned by this function
     can be used only while @file is open. When file is closed, this pointer
     is freed, so if you want to use it later, store it to another memory
     block.
   */
   l_ptr      (*linear) ( p_protocol p, l_protocolfile file );

   l_int      (*access)(p_protocol o, cl_text filename, l_int flags);
   l_int      (*chmod)(p_protocol o, cl_text path, l_int mode );
   l_int      (*mkdir)(p_protocol o, cl_text path, l_int mode );
   l_int      (*rmdir)(p_protocol o, cl_text dirname );
   l_int      (*rename)(p_protocol o, cl_text oldname, cl_text newname);
   l_int      (*remove)(p_protocol o, cl_text file);

   /*
     Compares 2 files, returned values like in string_compare
   */
   l_int      (*compare)(p_protocol o, cl_text file1, cl_text file2);

} t_protocol;

extern  p_protocol def_protocol;

#define DEF_PROTOCOL_FUNC(y) DEF_FUNC(def_protocol,y)
#define PROTOCOL(o) ((p_protocol)(o))

extern p_protocol  protocol_init ( p_protocol o, l_text prefix );
extern p_protocol  protocol_free ( p_protocol o );


extern p_vector    protocols;

p_protocol  protocols_get ( l_text filename );
p_protocol  protocols_add ( p_protocol protocol );
void        protocols_remove ( p_protocol x );
void        protocols_remove_all ( void );

#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xprotocl.h */


