/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XRESFILE.H                                    */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of MatrixOS                                */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XRESFILE_H_INCLUDED__
#define __XRESFILE_H_INCLUDED__

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#include"basics.h"
#include"drvbas.h"

/*
  --RESOURCE--
  --CHUNK--Normal Text--DATA--Hello Text--BODY--Hello world--DATA--How are you--DATA--BlaBlaBla
  --END OF RESOURCE--
*/

#define  RES_DATA         "--DATA--"
#define  RES_BODY         "--BODY--"
#define  RES_TITLE        "--TITLE--"
#define  RES_CHUNK        "--CHUNK--"
#define  RES_BEGIN        "--RESOURCE--"
#define  RES_END          "--END OF RESOURCE--"
#define  RES_BEGINL       12
#define  RES_ENDL         19

#define  RES_ICON         "Icon"
#define  RES_BITMAP       "Bitmap"
#define  RES_TEXT         "Text"
#define  RES_CURSOR       "Cursor"
#define  RES_ANI          "Ani"

/*
   Each Resource file start with --RESOURCE-- and end with --END OF RESOURCE--
   Between these two definitions is placed resource file of many chunks.
   Each --CHUNK-- separates different data types and follows by the definition
   of data style in text format.
   Then must folows --DATA-- that separate 1 or more data of the same CHUNK
   style.
   After --DATA-- *CAN* be the title of this DATA that's unique for the
   data of this CHUNK style. There can't be 2 same title of the same
   DATA style.
   Then can follow by --BODY-- depended if the title was defined or not.
   If the title wasn't previously defined, --BODY-- is optional. Then can
   follows more body that will be later indexed as 0,1,2,....
*/
typedef struct t_resourse *p_resource;
typedef struct t_resourse {

  l_text       definition;
  l_text       title;
  l_ptr        ptr;

} t_resource;

/*
  In this format is returned bitmap from resources.
*/
typedef struct t_res_bitmap *p_res_bitmap;
typedef struct t_res_bitmap {

  /* width of the bitmap */
  l_long  w;
  /* height of the bitmap */
  l_long  h;
  /* size of the bitmap */
  l_long  size;
  /* x-hot point - important in cursor*/
  l_long  xhot;
  /* y-hot point - important in cursor */
  l_long  yhot;
  /* information about bitmap in 24 bpp. RGB, RGB, ...*/
  /* 1.. is increased when important */
  l_byte  dat[1];

} t_res_bitmap;


/*
  In this format is returned cursor from resources.
*/
typedef struct t_res_bitmap *p_res_cursor;
typedef struct t_res_bitmap  t_res_cursor;

/*
  In this format is returned icon from resources.
*/
typedef struct t_res_bitmap *p_res_icon;
typedef struct t_res_bitmap  t_res_icon;


/*
  Parse the text resource to the vector of t_resource structure.
  @txt - is the text that contains resource information from --RESOURCE--
         to --END OF RESOURCE--.
         @txt is just the text before these two definitions in the file.
         @callback is used in the case of script where data after --BODY--
         is the script text or function that do something. This function is
         then call by 3 arguments text after --BODY-- and must return newmemory
         block to something what was the result of this text ( function )
  @definition - definition that will only be read from resource,if NULL
                all definition are accepted

         1. argument - definition
         2. argument - data
         3. argument - new definition of data type, that's returned

  @my_title   - title that will load in definition. If NULL, all titles
                are supported.

*/

p_vector  resource_parse_all ( p_file f, l_ptr (*callback)(), l_text definition, l_text my_title );



/*
  Translate the text resource to the new resource file.
  @txt - is the text that contains resource information from --RESOURCE--
         to --END OF RESOURCE--.
         @txt is just the text before these two definitions in the file.
         @callback is used in the case of script where data after --BODY--
         is the script text or function that do something. This function is
         then call by 4 arguments : text after --BODY--, and pointer to
         l_long that will return in this field the size of returned
         memory block.

         l_ptr callback ( l_text definition, l_text text_after_body,
                          l_long *size_or_returned_memory,
                          l_text *new_returned_data_type_definition )

  Returns size of data sucessfully wrtiten into the @dst file.
*/
l_long  resource_translate ( l_text txt, l_ptr (*callback)(), p_file dst );


/*
  Converts vector of resources to the new vector that will be returned.
  For converting use the function that read ptr and return new ptr that
  has own memory.

  @callback - converting function
  @my_free  - function that can release memory of element returned
              by callback.

  Usage :

  p_vector  p = resource_convert(icon_resource, &xgrx_conv_to_system_icon, low_grx_destroy_bitmap)

  (p) now contains all from resource, where each element is
  in the system depended bitmap.

Note :

  If you call this function, be sure sure, that each element in the vector
  (res) really contains information that can be converted by function
  callback.

  If you are not sure, please prefer function "resource_convert_as" that
  will look for your definition in resources.
*/
p_vector  resource_convert ( p_vector res, l_ptr (*callback)(l_ptr in), void (*my_free)(l_ptr out) );


/*
  Converts vector, that has elements of definition ( @definition )
  to the new vector that will be returned.
  For converting use the function that read ptr and return new ptr that
  has own memory.

  @callback - converting function
  @my_free  - function that can release memory of element returned
              by callback.

  Usage :

  p_vector  p = resource_convert_as(icon_resource, "Icon", &xgrx_conv_to_system_icon, low_grx_destroy_bitmap)

  (p) now contains all icons from resource, where each element is
  in the system depended bitmap.
*/
p_vector  resource_convert_as ( p_vector res, l_text definition, l_ptr (*callback)(l_ptr in), void (*my_free)(l_ptr out) );


/*
  Function converts resources to vector of vectors.

  It looks for all definitions in resource and for each title in the
  definition makes own vector that may contains one or more elements
  of the same title and the same defintion.

  This function is used for example in icons, where one icon in resources,
  may have more size of icons. So the title of all is the same and all
  in the vector are indexed from 0..length of vector.

  If you see how resource works, you will find out that one resource
  may have few definitions ( chunks ),
  each definition ( chunk ) may have more titles and each title may
  have more bodies. This function looks for all chunks in resources and
  search shunk that was called @definition. For this chunk loop all titles
  and for all titles save vector of bodies and convert data in the body
  by function @callback, and can free it by @my_free. For more information
  about these 2 functions, please see "resource_convert_as" and
  "resource_convert" functons.

  Returns vector that contains elements, where the each element
  contains data, that was converted by function @callback and can be released
  by function @my_free.

  It's useful in icons, where you need to store information about
  all icons in one vector and need to store information about each bitmap
  in alone element. Please note that icons usually have more than one
  bitmaps ( e.g. 2 for 32x32 and 16x16 ).

  Usage :

  p_vector  p = resource_convert_to_deflib(icon_resource, "Icon", &xgrx_conv_to_system_icon, low_grx_destroy_bitmap)

  Now when you need to get 32x32 bitmap of your first icon, call...,

  vector_element(vector_element(p, 0), 0);
                 |                     |
                 - first icon          - first bitmap
                                        ( 32x32...see xexec.h why 32x32 is the first in order )

  or 16x16 bitmap of your first icon

  vector_element(vector_element(p, 0), 1);
                 |                     |
                 - first icon          - second bitmap
                                        ( 16x16...see xexec.h why 16x16 is the second in order )

  vector_remove_all(p);
*/
p_vector  resource_convert_to_deflib ( p_vector res, l_text definition, l_ptr (*callback)(l_ptr in), void (*my_free)(l_ptr out) );


/*
  Returns pointer from vector that contains structures t_resource.
  E.g.
  If @res owns 5 chunks of t_resource where each has the same
  definition = "Text" ... and we try this function as :
  resource_get(@res, "Text", 3), this function returns 3rd t_resource->ptr
  from the vector.
*/
l_ptr     resource_get_at ( p_vector res, l_text definition, l_int index );
l_ptr     resource_get_definition_at ( p_vector res, l_int index );


/*
  Returns pointer from vector that contains structures t_resource.
  E.g.
  If @res owns 5 chunks of t_resource where each has the same
  definition = "Text" ... and we try this function as :
  resource_get(@res, "Text", "Title", 3), this function returns t_resource->ptr
  of t_resource, where t_resource->definition = "Text" and
  t_recource->title = "Title" placed at the @index positions of bodies
  from the vector.
*/
l_ptr     resource_get_as_at ( p_vector res, l_text definition, l_text title, l_int index );
l_ptr     resource_get_definition_as_at ( p_vector res, l_text title, l_int index );



/*
  Reads file filename and return vector of t_resource structs.
  Each struct contains definition name and ptr. This pointer is
  pointer to data which recognizes to definition style.
  E.g.
  definition = TEXT
  data = "Hello"

  definition = BITMAP
  data =

      0..4   with   of bitmap
      4..8   height of bitmap
      10..   bitmap ... each bitmap must be in 24bits depth where each
             pixel has 3 bytes of RGB.
*/

p_vector  resource_get ( l_text filename );
p_vector  resource_get_definition ( l_text filename, l_text definition );


/*
  Translate resource script and make the new one resource.

  @Callback is the same as in the resource_translate function. See for more
  information.
*/
l_long  resource_convert_script ( l_text filename, l_text newfile, l_ptr (*callback)());



#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xresfile.h */

