/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XSTRING.H                                     */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of MatrixOS                                */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XSTRING_H_INCLUDED__
#define __XSTRING_H_INCLUDED__

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#include"basics.h"
#include"drvbas.h"

static l_char  CSTRING_BUFFER[BUFFER_FORMAT_TEXT];


typedef struct t_string_buffer *p_string_buffer;
typedef struct t_string_buffer {

        l_long  block;
        l_long  wrote;
        l_text  text;

} t_string_buffer;


typedef struct t_string_tokenizer *p_string_tokenizer;
typedef struct t_string_tokenizer {

   l_text   text;
   l_text   token;
   l_text   tokenhelp;
   l_text   lasttoken;
   l_text   sep;

} t_string_tokenizer;


/*
  Returns the first occurence of character in string @s that not equals
  to one of the characters in the @chars string.
  E.g.

  l_text out = string_to_first_nonignore_chars ( "==Hello", "-,=");

  output :

  out = pointer to first occurence of 'H' in @s.
*/
l_text   string_to_first_nonignore_chars ( l_text s, l_text chars );


/*
  Returns the first occurence of character in string @s that equals
  to one of the characters in the @chars string.
  E.g.

  l_text out = string_to_first_nonignore_chars ( "Hello==Hi", "-,=");

  output :

  out = pointer to first occurence of '=' in @s.

*/
l_text   string_to_first_ignore_chars ( l_text s, l_text chars );


/*
  Returns the new memory of the string @s BUT without all characters
  placed in @chars.

  E.g.
  l_text out = string_ignore_chars("H-e-l-l-o , w=o=r=l=d  ", "- =");

  Ouput :
         Hello,world

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text   string_ignore_chars ( l_text s, l_text chars );


/*
  Returns the new memory of the string @s BUT without all space characters.

  E.g.
  l_text out = string_ignore_chars("Matrix  OS\n");

  Ouput :
         MatrixOS

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text   string_ignore_spaces ( l_text s );


/**
 * Returns TRUE if character @c is one of string chars @chrs.
*/
l_bool  string_is_one_of ( l_char c, l_text chrs );


/**
 * Returns new string without of white spaces on both sides.
 *
 * NOTE :
 * The memory returned by this call must be freed by the caller.
*/
l_text  string_trim ( l_text s );


/*
  Returns the first occurence of character in string @s that not equals
  to any space characters.
  E.g.

  l_text out = string_to_first_nonignore_spaces ( "        \nHello");

  output :

  out = pointer to first occurence of 'H' in @s, it means it ignores all
        characters before Hello.
*/
l_text   string_to_first_nonignore_spaces ( l_text s );


/*
  Returns the length of string. If string is NULL, it returns NULL
*/
l_long  string_length ( l_text s );


/*
  This function returns a pointer to the first occurrence of C in S.
  Note that if C is `NULL', this will return a pointer to the end of the
  string.

  Returns a pointer to the character, or `NULL' if it wasn't found.
*/
l_text  string_searchchar ( l_text s, l_char ch );


/*
  The same as string_searchchar, but it is NOT case sensitive.

  This function returns a pointer to the first occurrence of C in S.
  Note that if C is `NULL', this will return a pointer to the end of the
  string.

  Returns a pointer to the character, or `NULL' if it wasn't found.
*/
l_text  string_isearchchar ( l_text s, l_char ch );


/*
  This function finds the first occurrence of S2 in S1.

  Returns a pointer within S1, or `NULL' if S2 wasn't found.
*/
l_text  string_search ( l_text s, l_text d );


/*
  The same as string_search, but it is NOT case sensitive.

  This function finds the first occurrence of S2 in S1.

  Returns a pointer within S1, or `NULL' if S2 wasn't found.
*/
l_text  string_isearch ( l_text s, l_text d );


/*
  Returns a newly allocated area of memory that contains a duplicate of
  the string pointed to by SOURCE.

  Returns the newly allocated string, or NULL if there is no more memory.

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text   string_dup ( l_text s );


/*
  Returns a new allocated area of memory that contains a duplicate of
  the string pointed to by SOURCE of maximum length of characters = @size.
  If the size is < 0, it returns the whole string ( the same as string_dup )

  Returns the newly allocated string, or NULL if there is no more memory.

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text   string_ndup ( l_text str, l_long size );



/*
  Returns the new string of formatted output from the @arguments.
  The buffer for the string can be maximum BUFFER_FORMAT_TEXT-1 length

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text  string_vformated ( cl_text format, VA_LIST arguments );


/*
  Returns the new string of formatted output from the @arguments and
  returned text can be in size to maxchars.

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text  string_vformated_max ( l_int maxchars, cl_text format, VA_LIST arguments );


/*
  Returns the new string of formatted output from the arguments (...).

  NOTE :
  The memory returned by this call must be freed by the caller.


  The format string contains regular characters to print, as well as
  conversion specifiers, which begin with a percent symbol. Each
  conversion speficier contains the following fields:

   *  an optional flag, which may alter the conversion:

    `-'
          left-justify the field.

    `+'
          Force a `+' sign on positive numbers.

    `space'
          To leave a blank space where a plus or minus sign would have
          been.

    `#'
          Alternate conversion - prefix octal numbers with `0',
          hexadecimal numbers with `0x' or `0X', or force a trailing
          decimal point if a floating point conversion would have
          omitted it.

    `0'
          To pad numbers with leading zeros.

   *  A field width specifier, which specifies the minimum width of the
     field.  This may also be an asterisk (`*'), which means that the
     actual width will be obtained from the next argument.  If the
     argument is negative, it supplies a `-' flag and a positive width.

   *  An optional decimal point and a precision.  This may also be an
     asterisk, but a negative argument for it indicates a precision of
     zero.  The precision specifies the minimum number of digits to
     print for an integer, the number of fraction digits for a floating
     point number (max for `g' or `G', actual for others), or the
     maximum number of characters for a string.

   *  An optional conversion qualifier, which may be `h' to specify
     `short', `l' to specify long ints, or `L' to specify long doubles.
     Long long type can be specified by `L' or `ll'.

   *  The conversion type specifier:

    `c'
          A single character

    `d'
          A signed integer

    `D'
          A signed long integer

    `e'
    `E'
          A floating point number (double or long double).  The
          exponent case matches the specifier case.  The representation
          always has an exponent.

    `f'
          A floating point number (double or long double).  The
          representation never has an exponent.

    `g'
    `G'
          A floating point number (double or long double).  The
          exponent case matches the specifier case.  The representation
          has an exponent if it needs one.

    `i'
          A signed integer.

    `n'
          The next argument is a pointer to an integer, and the number
          of characters generated so far is stored in that integer.

    `o'
          A unsigned integer, printed in base 8 instead of base 10.

    `p'
          A pointer.  This is printed with an `x' specifier.

    `s'
          A `NULL'-terminated string.

    `u'
          An unsigned integer.

    `U'
          An unsigned long integer.

    `x'
    `X'
          An unsigned integer, printed in base 16 instead of base 10.
          The case of the letters used matches the specifier case.

    `%'
          A single percent symbol is printed.
*/
l_text  string_formated ( cl_text format, ... );

/**
 * See @string_formated and @string_vformated_max for more information
*/
l_text  string_formated_max ( l_int maxchars, cl_text format, ... );


/*
  Puts "..." before or after formated text if the length of text is
  greater than size. If size is -size, then puts "..." the before text,
  otherwise after.
  Returns the new text.

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text  string_formated_nice ( l_int size, l_text format, ... );


/*
  Inserts the new string @src to destination @dest at the position
  @from and maximum lenght of characters = @size.

  Each time is returned the new memory of the new string and the old one
  is released by the function.

  If @from = -1, it means we want to append the new string at the end of
  the destination.

  If @size = -1, it means we want to insert the whole @src source string
  to the destination

  E.g.

  l_text d = string_dup("Mr.");
  d = string_insert(d, " Michal ", -1, -1);
  d = string_insert(d, "Stencl", -1, -1);
  d = string_insert(d, "Hello ", 0, -1);

  d = "Hello Mr. Michal Stencl"

  low_free_mem(d);

  NOTE :
  Use function by non-static @dest string, @str can be static.
*/
l_text  string_insert ( l_text dest, l_text str, l_long from, l_long size );


/*
  Removes characters in the string from position @from and size @size.
  Returned string is the same as @dst but may be other, because memory
  was reallocated. If @size is -1, the whole text from position @from
  is removed.
  Use function by non-static @dest string.

  E.g.

  l_text s = string_delete(string_dup("Hello wxyzorld"), 7, 3);

  s = "Hello world"

  _free(s);
*/
l_text  string_delete ( l_text dest, l_long from, l_long size );


/*
  Appends the string @s at the end of the string @p.

  Returns the same pointer as @p, but maybe other ! because function
  use string_insert function.

  Note :
  Use function by non-static @p string. @s can be static.
*/
l_text  string_append ( l_text p, l_text s );


/*
  Appends one character @chr at the end of the string @p.

  Returns the same pointer as @p, but maybe other ! because function
  use string_insert function.

  Note :
  Use function by non-static @p string.
*/
l_text  string_append_char ( l_text p, l_char chr );


/*
  Appends number @d at the end of the string @p.
  ( number is converted to string )

  Returns the same pointer as @p, but maybe other ! because function
  use string_insert function.

  Note :
  Use function by non-static @p string.
*/
l_text  string_append_double ( l_text p, l_double d );


/*
  Appends number @f at the end of the string @p.
  ( number is converted to string )

  Returns the same pointer as @p, but maybe other ! because function
  use string_insert function.

  Note :
  Use function by non-static @p string.
*/
l_text  string_append_float ( l_text p, l_float f );


/*
  Appends number @d at the end of the string @p.
  ( number is converted to string )

  Returns the same pointer as @p, but maybe other ! because function
  use string_insert function.

  Note :
  Use function by non-static @p string.
*/
l_text  string_append_int ( l_text p, l_int d );


/*
  Appends number @d at the end of the string @p.
  ( number is converted to string )

  Returns the same pointer as @p, but maybe other ! because function
  use string_insert function.

  Note :
  Use function by non-static @p string.
*/
l_text  string_append_long ( l_text p, l_long d );


/*
  This function compares @s1 and @s2.

  Returns Zero if the strings are equal, a positive number if @s1
  comes after @s2 in the ASCII collating sequense, else a negative number.
  If the negative number is -2, one or both of strings are NULL.
*/
l_int    string_compare ( l_text s1, l_text s2 );


/*
  This function compares the two strings, disregarding case.

  Returns Zero if the strings are equal, a positive number if @s1
  comes after @s2 in the ASCII collating sequense, else a negative number.
  If the negative number is -2, one or both of strings are NULL.
*/
l_int    string_icompare ( l_text s1, l_text s2 );


/*
  Returns new string that contains integer number @x.
  This function converts number @x to string and returns new string.

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text   string_value_int ( l_int x );


/*
  Returns new string that contains integer number @x.
  This function converts number @x to string and returns new string.

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text   string_value_long ( l_long x );


/*
  Returns new string that contains integer number @f.
  This function converts number @f to string and returns new string.

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text   string_value_float ( l_float f );


/*
  Returns new string that contains integer number @d.
  This function converts number @d to string and returns new string.

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text   string_value_double ( l_double d );


/*
  Returns new string that contains one characterr @c.
  This function converts character @c to string and returns new string.

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text   string_value_char ( l_char c );


/*
  Converts string @str to character.

  Returns the first character of string.
*/
l_char   string_to_char ( l_text str );


/*
  Converts string @str to l_double.
*/
l_double string_to_double ( l_text str );


/*
  Converts string @str to l_float.
*/
l_float  string_to_float ( l_text str );


/*
  Converts string @str to l_long.
*/
l_long   string_to_long ( l_text str );


/*
  Converts string @str to l_int.
*/
l_int    string_to_int ( l_text str );


/*
  Returns character from the string placed at @index
*/
l_int    string_char_at ( l_text str, l_long index );


/*
  Replace all characters @oldchr in the string by new character @newchr.
  Returns @str.
*/
l_text   string_replace ( l_text str, l_int oldchr, l_int newchr );


/*
  Concates string @dst with @str and adds the character @whatbed before
  both. Returns @dst or new pointer depended on low_mem_realloc function.

  NOTE :
  Use function by non-static @dest string, @str can be static.
*/
l_text   string_add_before2 ( l_text dst, l_text str, l_text whatbef );
l_text   string_add_before ( l_text dst, l_text str, l_char whatbef );


/*
  Returns pointer to last occurence of string @sub in string @str.
*/
l_text string_last ( l_text str, l_text sub );


/*
  Returns pointer to last occurence of string @sub in string @str,
  discarding case.
*/
l_text string_ilast ( l_text str, l_text sub );


/*
  Tests if the string @str starts with string @prefix.
*/
l_bool   string_starts_with ( l_text str, l_text prefix );


/*
  Tests if the string @str starts with string @prefix, disregarding case.
*/
l_bool   string_istarts_with ( l_text str, l_text prefix );


/*
  Tests if the string @str starts with string @sufix.
*/
l_bool   string_ends_with ( l_text str, l_text sufix );


/*
  Tests if the string @str ends with string @sufix, disregarding case.
*/
l_bool   string_iends_with ( l_text str, l_text sufix );


/*
  Returns sub-string of string @s from position @from to the end of the
  string.

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text   string_substring ( l_text str, l_long from );


/*
  Returns sub-string of string @s from position @from and size @sz. If
  the @sz is -1, the function is the same as "string_substring"

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text   string_substring2 ( l_text str, l_long from, l_long sz );


/*
  Releases memory of the string and returns NULL.
*/
l_text   string_free ( l_text str );



/* string_buffer

   String buffer is the most powerfull class for strings. If you need
   to use big strings, it's better to use this class, mainly when
   you need to increase strings often or gets the length of the string.
*/
static l_long  stringbuf_expand ( p_string_buffer p, l_long size );


/*
  Set the lenght of the string to new one. String will be truncated to
  size.
  Returns the length of new block. If you need to get the size
  of the string use string_length.
*/
l_long  stringbuf_set_length ( p_string_buffer p, l_long size );


/*
  Appends string @s at the end of string buffer. Returns size
  of string that was succesfull appended.
*/
l_long  stringbuf_append ( p_string_buffer p, l_text s );


/*
  Appends character @chr at the end of string buffer.

  Returns size of chr that was succesfull appended = 1.
*/
l_long  stringbuf_append_char ( p_string_buffer p, l_char chr );


/*
  Appends ( converts number to string ) number @d at the end of
  string buffer.

  Returns size of string that was succesfull appended.
*/
l_long  stringbuf_append_double ( p_string_buffer p, l_double d );


/*
  Appends ( converts number to string ) number @f at the end of
  string buffer.

  Returns size of string that was succesfull appended.
*/
l_long  stringbuf_append_float ( p_string_buffer p, l_float f );


/*
  Appends ( converts number to string ) number @d at the end of
  string buffer.

  Returns size of string that was succesfull appended.
*/
l_long  stringbuf_append_int ( p_string_buffer p, l_int d );


/*
  Appends ( converts number to string ) number @d at the end of
  string buffer.

  Returns size of string that was succesfull appended.
*/
l_long  stringbuf_append_long ( p_string_buffer p, l_long d );


/*
  Converts string buffer to string.

  NOTE :
  Don't use this function often in big strings because this copy all
  string placed in string_buffer to new location. In the case the string
  has 500K and is called 100X there is copied 500.000.000 characters.
  Not very fast, is it ? :)

  The memory returned by this call must be freed by the caller.
*/
l_text  stringbuf_to_string ( p_string_buffer p );


/*
  Converts string buffer of @size from position @from to new string

  NOTE :
  The memory returned by this call must be freed by the caller.
*/
l_text  stringbuf_to_string2 ( p_string_buffer p, l_long from, l_long size );


/*
  Removes (n)-characters from @index, where (n) is @size.

  Returns number of characters sucesfull removed.
*/
l_long  stringbuf_delete ( p_string_buffer p, l_long index, l_long size );


/*
  Inserts string @s at @index to string buffer. If the index is -1, string
  is inserted at the end of the string_buffer.

  Returns number of succesfull inserted bytes.
*/
l_long  stringbuf_insert ( p_string_buffer p, l_long index, l_text s );


/*
  Inserts character @chr at @index to string buffer.
  If the index is -1, the character is inserted at the end of
  the string_buffer.

  Returns number of succesfull inserted bytes = 1.
*/
l_long  stringbuf_insert_char ( p_string_buffer p, l_long index, l_char chr );


/*
  Inserts (converts number to string) number @d at @index to string buffer.
  If the index is -1, the number is inserted at the end of
  the string_buffer.

  Returns number of succesfull inserted bytes.
*/
l_long  stringbuf_insert_double ( p_string_buffer p, l_long index, l_double d );


/*
  Inserts (converts number to string) number @f at @index to string buffer.
  If the index is -1, the number is inserted at the end of
  the string_buffer.

  Returns number of succesfull inserted bytes.
*/
l_long  stringbuf_insert_float ( p_string_buffer p, l_long index, l_float f );


/*
  Inserts (converts number to string) number @d at @index to string buffer.
  If the index is -1, the number is inserted at the end of
  the string_buffer.

  Returns number of succesfull inserted bytes.
*/
l_long  stringbuf_insert_int ( p_string_buffer p, l_long index, l_int d );


/*
  Inserts (converts number to string) number @d at @index to string buffer.
  If the index is -1, the number is inserted at the end of
  the string_buffer.

  Returns number of succesfull inserted bytes.
*/
l_long  stringbuf_insert_long ( p_string_buffer p, l_long index, l_long d );


/*
  Returns character placed at @index in string_buffer or ZERO if the @index
  is out of range.
*/
l_char  stringbuf_char_at ( p_string_buffer p, l_long index );


/*
  Returns string of string_buffer. This string is the pointer to internal
  memory of string_buffer. Please use this function for getting string
  of the string_buffer instead of stringbuf_to_string.

  Note :
  Can't release returned string, because it's the internal pointer of
  string_buffer class.
*/
l_text  stringbuf_look ( p_string_buffer p );


/*
  Copies character @chr at @index to string_buffer.

  Returns character that was copyied or ZERO, if the index is out of range.
*/
l_char  stringbuf_set_char_at ( p_string_buffer p, l_long index, l_char chr );


/*
  Returns lenght of the string in string_buffer.
*/
l_long  stringbuf_length ( p_string_buffer p );


/*
  Creates new string_buffer. The initial size of memory for the
  buffer is 16 bytes. If the memory needs to be increased the old size
  is multiply by 2. You couldn't set length of the string each time.

  String buffer makes it as default.
*/
p_string_buffer stringbuf_new ( void );


/*
  Creates new string_buffer. The initial size of memory for the
  buffer is @initsize bytes. If the memory needs to be increased, the old size
  is multiply by 2. You couldn't set length of the string each time.

  String buffer makes it as default.
*/
p_string_buffer stringbuf_new2 ( l_long initsize );


/*
  Creates new string_buffer and copy the string to the buffer.
  The initial size of memory for the buffer is length of
  the string + 16 bytes.
  If the memory needs to be increased, the old size
  is multiply by 2. You couldn't set length of the string each time.

  String buffer makes it as default.
*/
p_string_buffer stringbuf_new3 ( l_text str );


/*
  Release context of string buffer and returns NULL.
*/
l_ptr   stringbuf_free ( p_string_buffer p );



/* string_tokenizer

   This class is very powerfull and very usefull for generating tokens
   in the string. The usage is very simple.

   At the first create string_tokenizer by one of string_tokenize_newX
   function. Then use loop of string_tokenizer_next_token and the last
   value of the string_tokenizer_next_token can get by function
   string_tokenizer_last_token.

   Example1 :

   p_string_tokenizer p;
   l_text str = string_dup("Hello  world how are you   I'm fine");

   makes tokens of @str that was separated by char ' '

   p = string_tokenizer_new3(str, ' ');

   while ( string_tokenizer_next_itoken(p) ) {

      printf("token:%s\n", string_tokenizer_last_token(p));

   };

   Output :

   token:Hello
   token:world
   token:how
   token:are
   token:you
   token:I'm
   token:fine


   Example2 :

   p_string_tokenizer p;
   l_text str = string_dup("<html><!--Wow-->Hello<body>world</body></html>");

   p = string_tokenizer_new2(str);

   search tokens of @str that can be separated by any of the chars

   while ( string_tokenizer_next_itoken(p, "<>!--") ) {

      printf("token:%s\n", string_tokenizer_last_token(p));

   };

   Output :

   token:html
   token:Wow
   token:Hello
   token:body
   token:world
   token:/body
   token:/html

*/

/*
  Creates new string tokenizer that will use space character ' ' in
  function "string_tokenizer_next_token" and
  "string_tokenizer_next_itoken" disregarding case.

  Functions :

  string_tokenize_new
  string_tokenize_new2
  string_tokenize_new3

  use different values only in the case you use simple function
  "string_tokenizer_next_token" or
  "string_tokenizer_next_itoken" disregarding case.

  In using "string_tokenizer_next_tokenX" or
  "string_tokenizer_next_itokenX" has all string_tokenizer_newX functions
  the same effect.
*/
p_string_tokenizer  string_tokenizer_new2 ( l_text s );


/*
  Creates new string tokenizer that will use character @chr in
  function "string_tokenizer_next_token" and
  "string_tokenizer_next_itoken" disregarding case.

  Functions :

  string_tokenize_new
  string_tokenize_new2
  string_tokenize_new3

  use different values only in the case you use simple function
  "string_tokenizer_next_token" or
  "string_tokenizer_next_itoken" disregarding case.

  In using "string_tokenizer_next_tokenX" or
  "string_tokenizer_next_itokenX" has all string_tokenizer_newX functions
  the same effect.
*/
p_string_tokenizer  string_tokenizer_new3 ( l_text s, l_char c );


/*
  Creates new string tokenizer that will use string @sep in
  function "string_tokenizer_next_token" and
  "string_tokenizer_next_itoken" disregarding case.

  Functions :

  string_tokenize_new
  string_tokenize_new2
  string_tokenize_new3

  use different values only in the case you use simple function
  "string_tokenizer_next_token" or
  "string_tokenizer_next_itoken" disregarding case.

  In using "string_tokenizer_next_tokenX" or
  "string_tokenizer_next_itokenX" has all string_tokenizer_newX functions
  the same effect.
*/
p_string_tokenizer  string_tokenizer_new ( l_text s, l_text sep );


/*
  Releases string tokenizer and returns NULL.
*/
l_ptr  string_tokenizer_free ( p_string_tokenizer p );


/*
  Returns the token from the current position. You can later get the
  same value by function "string_tokenizer_last_token".
  When you need to store the value please use string_dup, because the next
  calling of this function returns other value and your old value will be
  released by function.

  If there is no token, returns NULL.
*/
l_text   string_tokenizer_next_token ( p_string_tokenizer p );


/*
  Returns the token disregarding case from the current position.
  You can later get the
  same value by function "string_tokenizer_last_token".
  When you need to store the value please use string_dup, because the next
  calling of this function returns other value and your old value will be
  released by function.

  If there is no token, returns NULL.
*/
l_text   string_tokenizer_next_itoken ( p_string_tokenizer p );


/*
  Returns the last token. The same value as was returned by last calling
  of function "string_tokenizer_next_tokenX" or
  "string_tokenizer_next_itokenX" disregarding case.
*/
l_text   string_tokenizer_last_token ( p_string_tokenizer p );


/*
  Returns the token separated by string @sep from the current position.
  You can later get the
  same value by function "string_tokenizer_last_token".
  When you need to store the value please use string_dup, because the next
  calling of this function returns other value and your old value will be
  released by function.

  If there is no token, returns NULL.
*/
l_text   string_tokenizer_next_token2 ( p_string_tokenizer p, l_text sep );


/*
  Returns the token disregarding case separated by string @sep
  from the current position. You can later get the
  same value by function "string_tokenizer_last_token".
  When you need to store the value please use string_dup, because the next
  calling of this function returns other value and your old value will be
  released by function.

  If there is no token, returns NULL.
*/
l_text   string_tokenizer_next_itoken2 ( p_string_tokenizer p, l_text sep );


/*
  Returns the token separated by any of the characters in @chars
  from the current position. You can later get the
  same value by function "string_tokenizer_last_token".
  When you need to store the value please use string_dup, because the next
  calling of this function returns other value and your old value will be
  released by function.

  If there is no token, returns NULL.
*/
l_text   string_tokenizer_next_token4 ( p_string_tokenizer p, l_text chars );


/*
  Returns the token separated by character @chr
  from the current position. You can later get the
  same value by function "string_tokenizer_last_token".
  When you need to store the value please use string_dup, because the next
  calling of this function returns other value and your old value will be
  released by function.

  If there is no token, returns NULL.
*/
l_text   string_tokenizer_next_token3 ( p_string_tokenizer p, l_char chr );


/*
  Returns the token disregarding case separated by character @chr
  from the current position. You can later get the
  same value by function "string_tokenizer_last_token".
  When you need to store the value please use string_dup, because the next
  calling of this function returns other value and your old value will be
  released by function.

  If there is no token, returns NULL.
*/
l_text   string_tokenizer_next_itoken3 ( p_string_tokenizer p, l_char chr );


#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xstring.h */

