/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XVECTOR.H                                     */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of MatrixOS                                */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XVECTOR_H_INCLUDED__
#define __XVECTOR_H_INCLUDED__

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#include"basics.h"


typedef struct t_vector *p_vector;
typedef struct t_vector {

   l_ptr       ptr;
   void      (*free)(l_ptr);
   p_vector    next;

} t_vector;


/*
  Class vector is one of the most using classes in the system.
  It's an array that stores dynamic elemtents. When class is destroyed
  all elemtens are relesed too.

  Each vector contains information of the content @ptr and the @free
  function that can release the content @ptr and pointer to @next vector.
  If the vector has NULL as @next, it means it's last vector. The First
  is that, from which you want to start. Function vector_add still returns
  the first vector and appends the new one at the end of the old one.
*/

/*
  Makes new element with pointer @x and function @f that can release
  memory of @x and adds it to the array. Vectors haven't any contructors
  they use only vector_add by @arr NULL as the contructor.

  Returns pointer to new vector.

  E.g.

  p_vector v  = vector_add(NULL, string_dup("Hello"), low_mem_free);
  v  = vector_add(v, string_dup("world"), low_mem_free);
  v  = vector_add(v, string_dup("!"), low_mem_free);
  v  = vector_add(v, string_dup("I'm"), low_mem_free);
  v  = vector_add(v, string_dup("vector"), low_mem_free);

  l_text t = vector_element(v, 1);

  @t = "world"

  vector_remove_all(v);
*/
p_vector  vector_add ( p_vector arr, l_ptr x, void (*f)(l_ptr p) );


/**
 * This function collects vector @v by name, where each name of the
 * each text element is get by function "func_get_name", where @x is
 * pointer of the same type as 2nd argument in vector_add.
 * Returns the same vector as @v is.
*/
p_vector  vector_collect_by_name ( p_vector v, l_text (*func_get_name)(l_ptr x) );


/**
 * This function collects vector @v by name, where each name of the
 * each text element is get by function "func_get_name", where @x is
 * pointer of the same type as 2nd argument in vector_add.
 * Returns the same vector as @v is.
*/
p_vector  vector_collect_by_iname ( p_vector v, l_text (*func_get_name)(l_ptr x) );


/**
 * This function collects vector @v by name, where each name of the
 * each text element is get by function "func_get_name", where @x is
 * pointer of the same type as 2nd argument in vector_add.
 * Returns the same vector as @v is.
*/
p_vector  vector_collect_by_name_ex ( p_vector v, l_bool case_sens, l_text (*func_get_name)(l_ptr x) );


/**
 * This function collects vector @v by name, where each number of the
 * each text element is get by function "func_get_number", where @x is
 * pointer of the same type as 2nd argument in vector_add.
 * Returns the same vector as @v is.
*/
p_vector  vector_collect_by_number ( p_vector v, l_bool greatest_1st, l_big (*func_get_number)(l_ptr x) );


/*
  The same as "vector_add" function defined above, but the new vector is
  add at the first position and new one is returned.

  E.g.

  p_vector v  = vector_add(NULL, string_dup("Hello"), low_mem_free);
  v  = vector_add(v, string_dup("world"), low_mem_free);
  v  = vector_add(v, string_dup("!"), low_mem_free);
  v  = vector_add(v, string_dup("I'm"), low_mem_free);
  v  = vector_add_first(v, string_dup("vector"), low_mem_free);

  l_text t = vector_element(v, 0);

  @t = "vector"

  vector_remove_all(v);
*/
p_vector  vector_add_first ( p_vector arr, l_ptr x, void (*f)(l_ptr p) );


/*
  This may use as contructor. It's the same as "vector_add(NULL, x, f)".

  Returns new vector.
*/
p_vector  vector_add2 ( l_ptr x, void (*f)(l_ptr p) );


/*
  Appends vector @arr at the end of the vector @dst. If @dst not
  exists, returns @arr, otherwise returns @dst.
*/
p_vector  vector_add_vector ( p_vector dst, p_vector arr );


/*
  Returns vector at the index @at in vectors starting from @arr.
*/
p_vector  vector_at ( p_vector arr, l_long at );


/*
  Returns pointer of the vector at the index @at in vectors starting from
  vector @arr.
*/
l_ptr   vector_element ( p_vector arr, l_long at );


/*
  Returns position of the pointer @ptr in vectors starting from @arr.
  Opposite function to "vector_element".
  If not find the pointer @ptr returns -1, otherwise returns index of
  vector where @ptr is placed.
*/
l_long    vector_find ( p_vector arr, l_ptr ptr );


/*
  Tests if vectors starting from @arr
  contains @ptr. The same as  "vector_find(arr,ptr) != -1"
*/
l_bool    vector_contains ( p_vector arr, l_ptr ptr );


/*
  Returns number of vectors starting from vector @arr.
*/
l_long    vector_size ( p_vector arr );


/*
  Returns last vector starting from @arr.
*/
p_vector    vector_last ( p_vector arr );


/*
  Find pointer @ptr in vectors and in the case it was succesfull,
  removes the vector from @arr and release memory of @ptr by function
  @free. If @arr is the first vector, returns the next one, otherwise
  returns the first one.
*/
p_vector  vector_remove ( p_vector arr, l_ptr x );


/*
  Removes vector at index @w starting from vector @arr and releases
  the memory of @ptr in vector going to removed. If @w is ZERO,
  returns the second vector ( next from @arr ), otherwise returns
  the first one.
*/
p_vector  vector_remove_at ( p_vector arr, l_long w );


/*
  Removes all vectors starting from vector @arr and releases
  memories of all @ptr in vectors going to removed.

  Returns NULL.
*/
p_vector  vector_remove_all ( p_vector arr );


/**
 * Removes vectors that starts from position @from
*/
p_vector  vector_remove_from ( p_vector arr, l_long from );


/*
  Removes all vectors starting from vector @arr and releases
  memories of all @ptr in vectors going to removed if func with the
  pointer of the vector is true.

  Returns NULL.
*/
p_vector  vector_remove_all_if ( p_vector arr, l_bool (*func)(l_ptr) );


/*
  The same as vector_remove_all, but for the compatibility with vector
  free function is this definition too.

  Why do that ?
  Each vector may contains as @ptr new vector and so on, for example
  icons_system. Icon is the vector of more bitmaps 16x16,32x32 etc. and
  icons_system is the vector of more bitmaps.

  E.g.

  p_vector v  = vector_add(NULL, string_dup("Hello"), low_mem_free);
  v  = vector_add(v, vector_add(NULL, string_dup("world"), low_mem_free), &vector_remove_all);
  v  = vector_add(v, string_dup("!"), low_mem_free);
  v  = vector_add(v, string_dup("I'm"), low_mem_free);
  v  = vector_add(v, string_dup("vector"), low_mem_free);

  l_text t1 = (l_text)vector_element(v, 0);
  l_text t2 = (l_text)vector_element(vector_element(v, 1), 0);

  @t1 = "Hello"
  @t2 = "world"

  vector_remove_all(v);

  ...release also memory of string in v[1] and then the memory of the vector.
*/

void  vector_remove_all2 ( l_ptr arr );


/* first in first out */

/*
  Copies @src to @list at position @where, where each element has fixed
  size @size. If @list not exists, returns new one with the context
  @src and size @size, otherwise reallocate memory in size @size*(@where+1)

  Returns @list or if the list was NULL, or reallocated to other memory,
  returns new one.
*/
l_ptr fifo_add ( l_ptr list, l_ptr src, l_int size, l_int where );


/*
  Returns pointer of the data placed at index @where and size @size and
  decrease size of @list to @size*@where. If @where is ZERO, all memory
  is released and @list will returns NULL.

  Use this function only for small data chunks ( intengers, small classes )
  that not odd the 512 bytes for one chunk.
*/
l_ptr fifo_get ( l_ptr *list, l_int size, l_int where );


#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xvector.h */

