/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XVIEW.H                                       */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of QubeOS                                  */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XVIEW_H_INCLUDED__
#define __XVIEW_H_INCLUDED__

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#include"basics.h"
#include"drvbas.h"


/* drag mode settings in view dragging (moving/resizing)
   used in t_view.drag_mode. You can choose the combination of
   following values to make ability in function t_view.drag_view
*/
/* it's possible to drag and move it */
#define DM_DRAGMOVE           0x0000001
/* it's possible to drag and grow it */
#define DM_DRAGGROW           0x0000002
/* drag all context of window, not only the frame */
#define DM_DRAGCONTEXT        0x0000004
/* drag like the icon */
#define DM_DRAGICONIC         0x0000008


/** Drag & Drop definitions - used in view_drag_and_drop function in
 *  argument @mode.
*/
/* It's used by left mouse button */
#define DD_CUT                0x0001
/* It's used by right mouse button and function */
#define DD_OPTIONAL           0x0002


/* options settings
   used in t_object.option. For more options see file "xobject.h"
   Please do not increase definitions of the option in your inherited
   objects. These are internal values used for the engine. The is the
   posibility thay will be increased in next versions.
*/
/* make it visible. This is default setting of view_init function. */
#define VW_OF_VISIBILITY      0x0001000

/* view ignore owner's limitation, this settings is out as default.
   You can choose this obtion before inserting view to another view,
   and the result will uncliping of view in the owner view.

   Each view have virtual function t_view.size_limits that returns the
   limit area of view, where can be child views visible. If this settings
   is set, this limitation not apply this child.
*/
#define VW_OF_IGNORELIM       0x0002000

/* check the view for mouse doubleclick
*/
#define VW_OF_DBLCLICK        0x0004000

/**
*/
#define VW_OF_NOTINDEX        0x0008000

/**
 * View is disposed when the end of modal state is occured and view
 * wasn't inserted to owner before.
*/
#define VW_OF_DISPOSEMODAL    0x0010000


/* state settings
   used in t_object.state. For more options see file "xobject.h"
   Please do not increase definitions of the option in your inherited
   objects. These are internal values used for the engine. The is the
   posibility thay will be increased in next versions.
*/
/* mouse is in view
   If this state is select, you are sure the mouse is in view, also in
   the case this child hasn't the focus.
   All states are called by virtual function t_object.set_state
   ( see "xobject.h" ).
*/
#define VW_SF_MOUSEINVIEW      0x0000040

/**
 * This flags is used when view can be redraw until this flag is not clear.
*/
#define VW_SF_DISABLE_DRAW     0x0000080

/**
 * This flags is used when data are draged.
*/
#define VW_SF_DRAGDATA         0x0000100



/* draw mode settings
   used in t_view.draw_mode and set options of drawing. Use function
   t_view.set_draw_mode any time before calling t_view.draw_view when
   you need to change options of drawing. Available changes are :

   DWM_TESTSUBVIEWS
   DWM_ONLYTOBUFFER
   DWM_ONLYTOSCREEN
   DWM_CANTACCELMOVE

   other one are internals.

   Note : after drawing of view, please set the old values.
*/
/* clip sub views */
#define DWM_TESTSUBVIEWS      0x0000001
/* output is only to buffer */
#define DWM_ONLYTOBUFFER      0x0000002
/* output is only to screen */
#define DWM_ONLYTOSCREEN      0x0000004
/* disable accel moving */
#define DWM_CANTACCELMOVE     0x0000008
/* draw under icon */
#define DWM_ICONICDRAW        0x0000010
/* has own dc */
#define DWM_OWNDC             0x0000020
/* not set as prefer when execute_view is called */
#define DWM_MENUDRAW          0x0000040
/* will ignore neighbour in drawing - it could redraw it */
#define DWM_IGNORENEIGHBOUR   0x0000080

/* alignments
   Used in t_view.align while you need to align child view to a limits
   of parent view.
*/

/*
   Aligns view to the left of the parent view. It means the @x coords, that
   was selected in view initialization are used as delta from left of parent.
*/
#define  VW_ALIGN_LEFT       0x001
/*
   Aligns view to the top of the parent view. It means the @y coords, that
   was selected in view initialization are used as delta from top of parent.
*/
#define  VW_ALIGN_TOP        0x002
/*
   Aligns view to the right of the parent view. It means the @x coords, that
   was selected in view initialization are used as delta from right of parent.
*/
#define  VW_ALIGN_RIGHT      0x004
/*
   Aligns view to the bottom of the parent view. It means the @y coords, that
   was selected in view initialization are used as delta from bottom of parent.
*/
#define  VW_ALIGN_BOTTOM     0x008
/*
  Aligns view to the center of parent in X.
*/
#define  VW_ALIGN_CENTERX    0x010
/*
  Aligns view to the center of parent in Y.
*/
#define  VW_ALIGN_CENTERY    0x020
/*
  Aligns view to the center of parent in XY.
*/
#define  VW_ALIGN_CENTER     (VW_ALIGN_CENTERY|VW_ALIGN_CENTERX)
/*
   Sets the object to fixed width, when the parent is resized. Important
   only in the case of aligmnets defined above.
*/
#define  VW_ALIGN_FIXEDX     0x100
/*
   Sets the object to fixed height, when the parent is resized. Important
   only in the case of aligmnets defined above.
*/
#define  VW_ALIGN_FIXEDY     0x200
/*
   Used the t_view.relsize.y that are calculated each time when the
   parent is resized. t_view.relsize.y owns information of relative width.
*/
#define  VW_ALIGN_RELATIVEX  0x400
/*
   Used the t_view.relsize.y that are calculated each time when the
   parent is resized. t_view.relsize.y owns information of relative height.
*/
#define  VW_ALIGN_RELATIVEY  0x800


/**
 * Retcodes returned by "drag_where" function
*/
/**
 * Returns if data are accpeted by receiver
*/
#define DRAGF_ACCEPT          1
/**
 * Returns if data are unsupported by receiver, but there are some
 * data than can be accepted.
*/
#define DRAGF_UNSUPPORT      -1
/**
 * Returns, if no data are supported by receiver, so it can try to
 * drop this data to its owner.
*/
#define DRAGF_NONE            0


/* brush states

   Use in function t_view.background when view is drawn. This function
   has the efect only in case it's used in your (draw)' functions

   see t_brush
*/

/* default settings of t_brush. If the t_bruhs.background exists the
   background is patterned.
*/
#define BRUSH_PATTERN         0x0000000
/* background is stretched.
*/
#define BRUSH_STRETCH         0x0000001
/* this view own the image. So the image will be destroyed in t_view.done
   function.
*/
#define BRUSH_SELFIMG         0x0000002
/* makes bacground by using gradient from @t_brush.color to @t_brush.color2
*/
#define BRUSH_GRADIENT        0x0000004
/* sets the gradient is horizontal, otherwise will be vertical
*/
#define BRUSH_GRADIENT_HOR    0x0000008
/* center background
*/
#define BRUSH_CENTER          0x0000010

#define OUTBUFFER(o)          (*(o))

/* use in all t_view.translate_event functions ath the begining of the
   function. It tests if the view is visible, and if has the focus.
*/
#define RETVIEW(o,e)\
   SUB_CODE {\
   if ( !(o->state & OB_SF_VISIBLE) ||\
         ( e->type == EV_MOUSE &&\
         !VIEW(o)->is_mouse_in_view(VIEW(o))) ) return;\
   } END_CODE


/* structure of the brush used in function t_view.background
*/
typedef struct t_brush {

  /* brush states defined above as BRUSH_XXXX */
  l_long     state;
  /* color of background */
  l_color    color;
  /* background bitmap or pattern */
  p_bitmap   background;
  /* color2 ( used in gradient ) */
  l_color    color2;
  /* reserved for next versions */
  l_char     reserved[22];

} t_brush;


typedef struct t_view *p_view;

/* info board */
void   view_ini ( void );


/* able to drag all context of object */
extern l_bool  drag_context_ok;
/* enable accelaration of moving */
extern l_bool  accel_moving_ok;



typedef struct t_view {

  struct t_object   obclass;

  struct t_brush    brush;
  struct t_rect     bounds;
  struct t_point    origin;
  struct t_relpoint relsize;
  struct t_point    scroll;
  struct t_rect     clip;
  struct t_rect    *safe_clip;

  p_graphics        draw_buffer;
  p_graphics       *draw_out;
  p_font            font;
  l_int             cursor;

  l_text            info_text;
  l_int             align;

  l_color          *palette;

  l_int             draw_mode;
  l_int             drag_mode;
  l_long            draw_lock;

  l_int             tab_index;
  l_int             tab_current;

  l_dword (*execute_view) ( p_view o, p_view sub );

  p_view  (*get_top_view_under_mouse) ( p_view o );
  p_view  (*top_view) ( p_view o );
  void    (*show_info_board) ( p_view o );

  l_bool  (*set_drag_draw_mode) ( p_view o, t_rect r );
  p_data  (*drag_data) ( p_view o, p_view *dst, l_word mode );
  l_dword (*drop_data) ( p_view dst, p_view src, p_data rec, t_point where, l_int mode );

  l_bool  (*drag_and_drop) ( p_view o, l_word mode, p_event event );
  t_rect  (*drag_rect) ( p_view o );
  l_int   (*drag_where) ( p_view o, p_data rec, t_point where );

  void    (*set_palette) ( p_view o, l_int *palette );
  void    (*drag_view) ( p_view o, l_word mode, t_event *event );
  t_rect  (*drag) ( p_view o, l_word mode, t_event *event );
  t_point (*size_minimum) ( p_view o );
  t_rect  (*get_limits) ( p_view o );

  void    (*reset_align) ( p_view o );

  p_object  (*close) ( p_view o );
  p_object  (*close_as) ( p_view o, l_dword end_state );

  void    (*draw) ( p_view o );
  void    (*draw_view) ( p_view o );
  void    (*draw_sub_views) ( p_view o, p_view from, p_view to );

  void    (*set_mouse_cursor) ( p_view o );
  l_bool  (*is_mouse_in_view) ( p_view o );
  l_bool  (*is_mouse_in_top) ( p_view o );
  p_view  (*get_view_under_mouse) ( p_view o );
  void    (*lock_drawing) ( p_view o );
  l_bool  (*unlock_drawing) ( p_view o );
  l_bool  (*is_draw_lock) ( p_view o );
  t_rect  (*get_bounds) ( p_view o );
  t_point (*mouse_where) ( p_view o );
  l_bool  (*mouse_inside) ( p_view o, t_rect r );
  l_bool  (*is_visible) ( p_view o );

  p_graphics (*begin_paint) ( p_view o, t_point *p, t_rect rwhere );
  void       (*end_of_paint) ( p_view o, t_rect rwhere );

  void    (*draw_overlays) ( p_view o );
  l_bool  (*draw_mini_box) ( p_view o, p_graphics out, p_graphics buffer, t_rect rwhere );

  void    (*draw_under_view) ( p_view o );
  void    (*draw_under_rect) ( p_view o, t_rect r );
  void    (*draw_in_rect) ( p_view o, t_rect r );
  void    (*draw_me) ( p_view o );

  l_color (*get_color) ( p_view o, l_int cpos );

  l_bool  (*is_top_view) ( p_view o );
  void    (*move_accel) ( p_view o, t_rect from, t_rect to );
  void    (*move_view) ( p_view o, t_rect r );
  void    (*grow_view) ( p_view o, t_rect r );

  l_bool  (*refresh) ( p_view o );


  void    (*set_redraw) ( p_view o, l_bool enable );

  t_rect  (*get_local_extent) ( p_view o );

  t_rect  (*get_global_bounds) ( p_view o, t_rect r );
  t_rect  (*get_local_bounds) ( p_view o, t_rect r );
  t_point (*get_global_point) ( p_view o, t_point p );
  t_point (*get_local_point) ( p_view o, t_point p );

  t_rect* (*get_not_overlays) ( p_view o, t_rect rlocal, t_rect rgroup,
                                p_view p, p_view target, l_long *rnum,
                                t_rect **rout, l_bool sub, l_bool ws );

  void    (*show) ( p_view o );
  void    (*hide) ( p_view o );

  void    (*set_tab_index) ( p_view o, l_int index );
  l_int   (*get_tab_index) ( p_view o );
  void    (*remove_tab_index) ( p_view o );
  p_view  (*find_able_tab_index) ( p_view o, l_int tab );
  l_int   (*select_child) ( p_view o, l_int tab, l_bool greater );
  l_int   (*find_last_tab_index) ( p_view o );

  void    (*for_each_sub_view_set_state) ( p_view o, l_dword st, l_bool set );
  void    (*set_draw_mode) ( p_view o, l_int dm, l_bool set );
  l_bool  (*is_draw_mode) ( p_view o, l_int dm );

  t_rect  (*size_limits) ( p_view o );
  t_rect  (*get_cliped_extent) ( p_view o );

  void    (*set_bounds) ( p_view o, t_rect r );

  l_rect  (*width) ( p_view o );
  l_rect  (*height) ( p_view o );

  void    (*set_visible) ( p_view o, l_bool set );

  void    (*change_bounds) ( p_view o, t_rect nr );

  void    (*set_clip) ( p_view o, p_graphics out, t_rect r );
  void    (*background) ( p_view o, p_graphics out, t_rect r );

  void    (*set_clips) ( p_view o, t_rect r );
  void    (*reset_clips) ( p_view o );

  p_graphics  (*get_buffer) ( p_view o );

  p_view  (*get_under_view) ( p_view d );
  p_view  (*get_over_view) ( p_view d );
  p_view  (*get_left_view) ( p_view d );
  p_view  (*get_right_view) ( p_view d );

  l_bool  (*event_mouse) ( p_view o, l_dword state, t_point where );
  l_bool  (*event_mouse_enter) ( p_view o, l_dword state, t_point where );
  l_bool  (*event_mouse_leave) ( p_view o, l_dword state );
  l_bool  (*event_key) ( p_view o, l_int keycode, l_char ascii, l_int shifts );

} t_view;

#define VIEW(o)   ((p_view)(o))

extern p_view def_view;

#define DEF_VIEW_FUNC(y) DEF_FUNC(def_view,y)
#define DEF_VIEW_OBJECT_FUNC(y) DEF_FUNC(OBJECT(def_view),y)

#define reset_clip(o, out)  set_clip(o, out, o->clip)

/* view's functions */

extern p_view  (*view_init) ( p_view o, t_rect r );
p_view  _view_init ( p_view o, t_rect r );

void   _show_info_board ( p_view from, l_text text );
l_bool is_top_view_under_mouse_in_me ( p_view o );


#define RESET_CURSOR(o)\
SUB_CODE {\
    OBJECT(mouse)->state |= MO_SF_MOUSEMOVE;\
    event_set(event, EV_MOUSE, 0, OBJECT(o));\
    OBJECT(o)->put_event(OBJECT(o), event);\
} END_CODE

#define BLIT(src,dst,sx,sy,dx,dy,w,h)\
SUB_CODE {\
    l_bool iv = mouse->is_visible(mouse);\
    t_rect sc;\
    t_rect dc;\
    gr_get_clip(src, (&sc.a.x), (&sc.a.y), (&sc.b.x), (&sc.b.y));\
    gr_get_clip(dst, (&dc.a.x), (&dc.a.y), (&dc.b.x), (&dc.b.y));\
    gr_set_clip(dst, dx, dy, dx+w, dy+h);\
    gr_set_clip(src, sx, sy, sx+w, sy+h);\
    mouse->hide(mouse);\
    gr_blit_graphics(dst, src, sx, sy, dx, dy, w+1, h+1);\
    if ( iv ) mouse->show(mouse);\
    gr_set_clip(dst, dc.a.x, dc.a.y, dc.b.x, dc.b.y);\
    gr_set_clip(src, sc.a.x, sc.a.y, sc.b.x, sc.b.y);\
} END_CODE

#define TEST_SUB_VIEWS(o, x)\
SUB_CODE {\
    l_bool  tsv = VIEW(o)->is_draw_mode(VIEW(o), DWM_TESTSUBVIEWS);\
    VIEW(o)->set_draw_mode(VIEW(o), DWM_TESTSUBVIEWS, true);\
    ##x;\
    VIEW(o)->set_draw_mode(VIEW(o), DWM_TESTSUBVIEWS, tsv);\
} END_CODE

#define NOT_ACCEL_MOVE(o, x)\
SUB_CODE {\
    l_bool  tsv = VIEW(o)->is_draw_mode(VIEW(o), DWM_CANTACCELMOVE);\
    VIEW(o)->set_draw_mode(VIEW(o), DWM_CANTACCELMOVE, true);\
    ##x;\
    VIEW(o)->set_draw_mode(VIEW(o), DWM_CANTACCELMOVE, tsv);\
} END_CODE

#define VIEW_CLIPED(o, r) rect_cliped(VIEW(o)->get_cliped_extent(VIEW(o)), r)


#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xview.h */

