/****************************************************************************
*   Copyright 1999, Caldera Thin Client Systems, Inc.                       *
*   This software is licensed under the GNU Public License.                 *
*   See LICENSE.TXT for further information.                                *
*                                                                           *
*   Historical Copyright                                                    *
*                                                                           *
*   Copyright (c) 1985,1991,1992 Digital Research Inc.			    *
*   All rights reserved.						    *
*   The Software Code contained in this listing is proprietary to Digital   *
*   Research Inc., Monterey, California, and is covered by U.S. and other   *
*   copyright protection.  Unauthorized copying, adaption, distribution,    *
*   use or display is prohibited and may be subject to civil and criminal   *
*   penalties.  Disclosure to others is prohibited.  For the terms and      *
*   conditions of software code use, refer to the appropriate Digital       *
*   Research License Agreement.						    *
*****************************************************************************
*		      U.S. GOVERNMENT RESTRICTED RIGHTS			    *
*                    ---------------------------------                      *
*  This software product is provided with RESTRICTED RIGHTS.  Use, 	    *
*  duplication or disclosure by the Government is subject to restrictions   *
*  as set forth in FAR 52.227-19 (c) (2) (June, 1987) when applicable or    *
*  the applicable provisions of the DOD FAR supplement 252.227-7013 	    *
*  subdivision (b)(3)(ii) (May 1981) or subdivision (c)(1)(ii) (May 1987).  *
*  Contractor/manufacturer is Digital Research Inc. / 70 Garden Court /     *
*  BOX DRI / Monterey, CA 93940.					    *
*****************************************************************************
* $Header: m:/davinci/users//groups/panther/dsk/rcs/desktree.c 4.9 92/04/03 17:15:38 sbc Exp $
* $Log:	desktree.c $
 * Revision 4.9  92/04/03  17:15:38  sbc
 * WNODEs and PNODEs to fars, lots of other housekeeping
 * 
 * Revision 4.8  92/03/13  14:41:50  sbc
 * Merge in Keiko's changes required for Double Byte Character Support
 * 
 * Revision 4.7  92/03/12  13:57:24  rsf
 * Merge in RSF's changes for icons on desktop and (LONG) => (TREE).
 * 
 * Revision 4.6  92/02/19  15:55:23  sbc
 * Replace refs to G.a_trees[] with calls to rsrc_gaddr().
 * 
 * Revision 4.5  92/02/13  10:15:34  anderson
 * Couple of bug fixes to add/delete on saved tree.
 * 
 * Revision 4.4  92/02/11  11:56:02  system
 * FNODE lists cannot be shared between windows (f_obid, etc.) so opening a
 * tree for a second window means copying the existing FNODE list.  Bummer.
 * 
 * Revision 4.3  92/02/05  18:07:04  anderson
 * Massive changes to re-enable dynamic tree updating in context of trees saved
 * to disk.
 * 
 * Revision 4.2  92/01/31  16:36:36  sbc
 * change params for dos_sdta() to take FCB far * insteaad of LONG.
 * 
 * Revision 4.1  91/12/20  14:55:14  anderson
 * ViewMAX 3 baseline.  Converted TNODEs to FNODEs for 10-12K data savings.
 * Preserving built trees in startup directory on disk (will test for DRDOSCFG
 * in future rev).  Not yet doing dynamic tree updates, so any folder add/delete
 * /rename operation won't change the tree, and the tree won't be rebuilt from
 * scratch.  Use View:Refresh on tree windows for now.  There's still one bug
 * when both tree windows are displaying the same drive and one gets hidden.
 * 
 * Revision 3.1  91/08/19  16:38:46  system
 * ViewMAX 2 sources
 * 
Date	Who	SPR	Comments
----	---	---	--------
911016  K.H		Add supporting double byte character set. (#if DBCS)
910522	RSF		Set treetag for root.
910520	RSF		Exchanged values HIDE/SHOW KIDS. HIDE = '-' -> they're
			being shown. SHOW = '+' -> they're being hidden.
910516	RSF		Don't set treetag of root's parent ('cause it ain't
			got one.)
910516	WHF		Fix flashing TREEINFO dialog
910501	RSF		Merge in Heather's C-E tree work.
910408	RSF		Fix warning by making FAR_LONG constants long.

*****************************************************************************/

#include "shell.h"
#include "viewapps.h"

extern	WORD	DOS_ERR;
extern	WORD	DOS_AX;
extern	WORD	DOS_BX;

extern	GLOBES	G;

extern	BYTE	start_path[65];		/* path ViewMAX was run from */

MLOCAL WORD		level;
MLOCAL TREE		tree;
MLOCAL BOOLEAN		tree_showing;
MLOCAL BOOLEAN		deep_path;
MLOCAL FNODE far	*folder;
MLOCAL FNODE far	*prev_folder;
MLOCAL BOOLEAN		keepwalkin;

#define	TREE_FNAME	"viewtree.$$"	/* file name for saved tree info     */
#define BUFFNODES	100
#define	TBUF_LEN	(sizeof(tbuffer))

GLOBAL	WORD		thndl;		/* file handle for saved tree info   */
GLOBAL	BYTE		treefile[65];	/* path\filename for saved tree info */
GLOBAL	FNODE		tbuffer[BUFFNODES];


/****************************************************************
 *  Initialize parameters for the tree code on startup.
 ****************************************************************/
BOOLEAN tree_init(void)
{
    WORD	n;

    G.num_tnodes = NUM_TNODES;

    for (n = 0; n < 26; n++)		/* Point drive trees to nothing yet */
	G.t_drive[n] = (FNODE far *)NULL;
      
    return TRUE ;
    
} /* end tree_init() */

/****************************************************************
 *  Get rid of ALL temporary files that are holding saved trees.
 ****************************************************************/
void tree_deinit(void)
{
    BYTE	drv;

    if (thndl)			/* first close any open one */
	dos_close(thndl);
    
    for (drv = 'A'; drv <= 'Z'; drv++)
    {
	sprintf(treefile, "%s%s%c", start_path, TREE_FNAME, drv);
	dos_delete( (LONG)(BYTE far *)treefile );
    }

} /* end tree_deinit() */

/****************************************************************
 *  Delete any memory-resident AND any disk copy of a tree for
 *  the given drive.  
 ****************************************************************/
void	tree_dirty(BYTE drv)
{
    if (drv >= 0 && drv <= 'Z'-'A')
	G.t_drive[drv] = (FNODE far *)NULL;	/* local version */
    
    sprintf(treefile, "%s%s%c", start_path, TREE_FNAME, drv + 'A');
    dos_delete( (LONG)(BYTE far *)treefile );		/* disk version */
	
} /* end tree_dirty() */


#if 0	/* not used at the moment.... */
/****************************************************************
 *  Return how many drives have trees built for them, and which
 *  drives they are (NOTE: max. of two can be built at a time)
 ****************************************************************/
BYTE	trees_built(WORD *bltdrvs)
{
    BYTE	ii, total;
    
    total = 0;
    for (ii = 0; ii < 26; ii++)
    {
	if (G.t_drive[ii] != (FNODE far *)NULL)
	{
	    bltdrvs[total] = ii;
	    total++;
	}
    }

    return(total);

} /* end trees_built() */
#endif /* 0 */


/****************************************************************
 *  Recursive routine that walks the entire directory structure
 *  of a drive and builds a list of FNODES.
 ****************************************************************/
MLOCAL	void tree_walk( PNODE far * thepath, BOOLEAN end_draw)
{
	WORD	ret, cnt;
	BYTE	tmp_path[LEN_ZFNAME+LEN_PASSW];	/* Holds the current	*/
						/* directory + password */
	BYTE	c_string[10];

	if ( level<=3 )
	{
	  sprintf( c_string, "%ld", G.g_ndirs0);
	  tedinfo_set( tree, TREECNT, (BYTE far *)c_string );
	  if( !tree_showing )
	  {
	    show_hide(FMD_START, tree);
	    tree_showing = TRUE;
	  }
	  else
	    draw_fld( tree, TREECNT );
	}

	dos_sdta( (FCB far *)&G.g_fcbstk[level] );
	dos_sfirst( (char far *)"*.*", F_SUBDIR );
	
	if( !DOS_ERR ){	

	    while( !DOS_ERR && !(G.g_fcbstk[level].fcb_attr & F_SUBDIR) )
		dos_snext();

	    if( !DOS_ERR ){
	      do
	      {
		if ((G.g_fcbstk[level].fcb_attr & F_SUBDIR) &&
		    (G.g_fcbstk[level].fcb_name[0]!='.'))
		{
		    fstrcpy(folder->f_name, 
				    (BYTE far *)G.g_fcbstk[level].fcb_name );
		    fstrcpy((BYTE far *)tmp_path, folder->f_name);
		    folder->f_attr = F_SUBDIR;
		    folder->f_time = G.g_fcbstk[level].fcb_time;
		    folder->f_date = G.g_fcbstk[level].fcb_date;
		    folder->f_size = 0x0L;
		    SET_TLVL(folder, level);
		    SET_ISTREE(folder, TRUE);
		    folder->f_next = (FNODE far *)NULL;

		    G.g_ndirs0++;

		    prev_folder = folder;
		    if ((folder = fn_alloc()) == (FNODE far *)NULL)
		    {
			/* don't use this alert or close treewin */
			/* if coming from Find */
			if (end_draw)
			{
			    graf_mouse(M_ON, 0x0L);
			    alert(1, EROBJMIS);
			    graf_mouse(M_OFF, 0x0L);
			}
			keepwalkin = FALSE;
			return;
		    }
		    prev_folder->f_next = folder;

		    cnt = 0;
		    do
		    {
		        dos_chdir( (LONG)(BYTE far *)tmp_path );
		        passw_strip( (LONG)(BYTE far *)tmp_path );
			if( DOS_ERR && DOS_AX==0x56 )
			{
			   ret=passw_perror( (BYTE far *) tmp_path, cnt++,
				   end_draw );
			   tree_showing = FALSE;	/* rdrw dialog */
			}
		    }while( ret && (DOS_ERR && DOS_AX==0x56) );

		    if (!DOS_ERR && level < MAX_LEVEL-1)
		    {
		    	level++;
		        tree_walk(thepath, end_draw);
		    }
		    
		    /* ignore anything that's too deep and continue on */
		    if ((level == MAX_LEVEL-1) && keepwalkin)
		    {
			dos_sdta( (FCB far *)&G.g_fcbstk[MAX_LEVEL] );
			dos_sfirst( (char far *)"*.*", F_SUBDIR);
			do 
			{
			    dos_snext();
			    if ((G.g_fcbstk[MAX_LEVEL].fcb_attr & F_SUBDIR) &&
				(G.g_fcbstk[MAX_LEVEL].fcb_name[0] != '.'))
				deep_path = TRUE;
			} while (!DOS_ERR);
			dos_chdir( (LONG)(BYTE far *)".." );
		    }
		}
		
		if (keepwalkin)
		{
		  dos_sdta( (FCB far *)&G.g_fcbstk[level] );
		  do
		  {
		      dos_snext();
		  }while(!DOS_ERR && !(G.g_fcbstk[level].fcb_attr & F_SUBDIR));
	        }

	      } while (keepwalkin && 
		       !DOS_ERR && 
		       (G.g_fcbstk[level].fcb_attr & F_SUBDIR));
	      
	   }
	}

	dos_chdir( (LONG)(BYTE far *)".." );
	level--;
	return;
	
} /* end tree_walk() */

/****************************************************************
 * Build a drive tree. 
 * Return FALSE if we ran out of FNODEs, TRUE otherwise.
 ****************************************************************/
MLOCAL	WORD tree_build( PNODE far * thepath, BYTE drv, BOOLEAN end_draw)
{
	G.g_ndirs0 = 0x0L;	
					/* Start the tree off */
	if ((folder = fn_alloc()) == (FNODE far *)NULL)
	{
	    /* don't use this alert or close treewin if coming from Find */
	    if (end_draw)
		alert(1, EROBJMIS);
	    return(FALSE);		/* Can't allocate any nodes */
	}

	thepath->p_flist = G.t_drive[drv] = folder;
	
	SET_TLVL(folder, 0);		/* set up the root node */
	SET_ISTREE(folder, TRUE);	
	folder->f_name[0] = drv+'A';
	fstrcpy( &folder->f_name[1], (char far *)":\\" ) ;

	prev_folder = folder;

	if ((folder = fn_alloc()) == (FNODE far *)NULL)
	{
	    fn_free(prev_folder);
	    G.t_drive[drv] = (FNODE far *)NULL;
	    /* don't use this alert or close treewin if coming from Find */
	    if (end_draw)
		alert(1, EROBJMIS);
	    return(FALSE);		/* Can't allocate 2nd node */
	}

	prev_folder->f_next = folder;

	rsrc_gaddr( R_TREE, TREEINFO, &tree ) ;
                                        /* keep them informed now and then */
	tree_showing = FALSE;		/* has whole tree been displayed? */
	graf_mouse(M_OFF, 0x0L);

	level = 0;
	deep_path = FALSE;
	keepwalkin = TRUE;
	tree_walk(thepath, end_draw);
	
	if (end_draw)
		show_hide( FMD_FINISH, tree );
	graf_mouse(M_ON, 0x0L);

	/* Throw away an incomplete tree, otherwise it'll never be able */
	/* to be fully built should more FNODEs become available later. */
	if (!keepwalkin)
	{
	    tree_dirty(drv);
	    return(FALSE);
	}
	else 
	{
	    if (deep_path)			/* warn of deep paths */
		alert(1, ERDEEPTR);
	
	    prev_folder->f_next = (FNODE far *)NULL;
	    fn_free(folder);
	    tree_parse(thepath);
	    G.g_ndirs0 = thepath->p_count;
	    return(TRUE);
	}
	
} /* end tree_build() */

/****************************************************************
 *  Return the number of FNODEs in the given FNODE list.
 ****************************************************************/
MLOCAL WORD tree_size(FNODE far *pf)
{
    WORD	count = 0;
    
    if (pf != (FNODE far *)NULL)
	do 
	{
	    count++;
	} while ((pf = pf->f_next) != (FNODE far *)NULL);
    
    return(count);
    
}

/****************************************************************
 *  Save the given tree away to disk.  Each drive is saved in its
 *  own file.  File contains a node count followed by the tree's
 *  FNODE list.
 *  NOTE: this routine assumes pflist isn't a null ptr.
 ****************************************************************/
BOOLEAN	tree_save(FNODE far *pflist)
{
    WORD	node_count, idx;
    WORD	listsize, writesize;
    WORD	write_err = FALSE;
    LONG	total, avail;
    
    node_count = tree_size(pflist);
    listsize = node_count * sizeof(FNODE);
    
    /* First make sure there's enough disk space for the proposed file. */
    dos_space(start_path[0] - 'A' + 1, &total, &avail);
    if (avail < (sizeof(WORD) + listsize))
	return(FALSE);
    
    /* Establish a unique path-filename for this drive's saved tree info. */
    sprintf(treefile, "%s%s%c", start_path, TREE_FNAME, pflist->f_name[0]);
    
    /* This drive may already have a file but create a fresh one each time. */

    thndl = dos_create( (LONG)(BYTE far *)treefile, F_HIDDEN);
#if 0
    thndl = dos_create( (LONG)(BYTE far *)treefile, 0);
#endif /* 0 */
    if (DOS_ERR)
	return(FALSE);
	
    dos_write(thndl, sizeof(WORD), (LONG)(WORD far *)&node_count );
    write_err |= DOS_ERR;    

    /*  To make sure the FNODEs are contiguous in memory, we must
     *  pass them through a local buffer before writing them out.
     */
    folder = pflist;
    while (listsize)
    {
	writesize = (listsize < TBUF_LEN) ? listsize : TBUF_LEN;

	for (idx = 0; idx < writesize / sizeof(FNODE); idx++)
	{
	    fmemcpy((void far *)&tbuffer[idx], (void far *)folder,
			sizeof(FNODE));
	    folder = folder->f_next;
	}

	dos_write(thndl, writesize, (LONG)(BYTE far *)tbuffer );
	write_err |= DOS_ERR;
	
	listsize -= writesize;
    }
			    
    dos_close(thndl);
    
    if (write_err)
    {
	dos_delete( (LONG)(BYTE far *)treefile );	/* don't trust file contents! */
	return(FALSE);
    }
    else return(TRUE);
		    
} /* end tree_save() */
    
/****************************************************************
 *  Crack out a given amount of saved tree from the deep-freeze
 *  or a tbuffer[]'s worth, whichever is less.
 *  Saved tree is given by the current thndl (assumed active),
 *  at the current file seek pointer.  
 *  Argument listsize = how much is left to read from tree,
 *  which is updated to reflect amount just read.
 *  Returns number of FNODEs read.
 ****************************************************************/
MLOCAL WORD tree_thaw(WORD *listsize)
{
    WORD	readsize;
    
    readsize = (*listsize < TBUF_LEN) ? *listsize : TBUF_LEN;

    if (dos_read(thndl, readsize, (LONG)(BYTE far *)tbuffer ) != readsize)
	return(FALSE);
    if (DOS_ERR)
	return(FALSE);

    *listsize -= readsize;
    return(readsize / sizeof(FNODE));
    
} /* end tree_thaw() */

/****************************************************************
 *  Find out whether a tree for the given drive is saved away.
 *  If so, open the file and record the file handle.  Otherwise
 *  return FALSE.
 ****************************************************************/
MLOCAL BOOLEAN	tree_issaved(BYTE drv)
{
    /* Use the unique path-filename for this drive's saved tree info. */
    sprintf(treefile, "%s%s%c", start_path, TREE_FNAME, drv);
    
    thndl = dos_open( (LONG)(BYTE far *)treefile, READWRITE);
    if (DOS_ERR)
	return(FALSE);

    return(TRUE);
    
} /* end tree_issaved() */
    
/****************************************************************
 *  Restore the given drive's tree from the deep-freeze.
 *
 * NOTE: BYTE f_junk, WORD f_wid, WORD f_obid, *f_pa, WORD f_isap
 * aren't used by trees....better not to save them to disk!
 ****************************************************************/
MLOCAL BOOLEAN	tree_restore( PNODE far * thepath, BOOLEAN end_draw)
{
    WORD	node_count;
    WORD	idx;
    WORD	listsize, readsize;
    BOOLEAN	root;
    
    if (!tree_issaved( thepath->p_spec[0] ))
	return(FALSE);
    
    dos_read(thndl, sizeof(WORD), (LONG)(WORD far *)&node_count );

    if (node_count == 0)
    {
	dos_close(thndl);
	return(FALSE);
    }
    
    root = TRUE;
    listsize = node_count * sizeof(FNODE);
    while (listsize)
    {
	if ((readsize = tree_thaw(&listsize)) == 0)
	{
	    dos_close(thndl);
	    return(FALSE);				/* file read error */
	}

	for (idx = 0; idx < readsize; idx++)
	{
	    /* Ignore deleted FNODEs by looking for an empty f_name[] field. */
	    if (tbuffer[idx].f_name[0] == '\0')
		continue;
	    
	    if ((folder = fn_alloc()) == (FNODE far *)NULL)
	    {
		/* don't use this alert or close treewin if coming from Find */
		if (end_draw)
		    alert(1, EROBJMIS);
		dos_close(thndl);
		return(FALSE);		/* Can't allocate any nodes */
	    }
	    
	    if (root)
	    {
		thepath->p_flist = folder;
		root = FALSE;
	    }
	    else prev_folder->f_next = folder;
		
	    fmemcpy((void far *)folder, (void far *)&tbuffer[idx],
			sizeof(FNODE));
	    
	    prev_folder = folder;
	    
	} /* for */
	
    } /* while */
    
    folder->f_next = (FNODE far *)NULL;
    
    dos_close(thndl);
    thepath->p_count = tree_size(thepath->p_flist) - 1;		/* not root */
    return(TRUE);
    
} /* end tree_restore() */

/****************************************************************
 *  Build a copy of the given tree in a new FNODE list and plug
 *  all it's info into the given path node.
 ****************************************************************/
MLOCAL BOOLEAN tree_copy( PNODE far * thepath, FNODE far *tree, BOOLEAN end_draw)
{
    FNODE far	*srcnode;
    FNODE far	*dstnode;
    WORD 	ii;
    
    fl_free(thepath->p_flist);		/* in case it had some other tree */

    srcnode = tree;
    for (ii = 0; (srcnode != (FNODE far *)NULL); srcnode = srcnode->f_next, ii++)
    {
	if ((dstnode = fn_alloc()) == (FNODE far *)NULL)
	{
	    /* don't use this alert or close treewin if coming from Find */
	    if (end_draw)
		alert(1, EROBJMIS);
	    break;
	}

	if (ii == 0)
	    thepath->p_flist = dstnode;
	else prev_folder->f_next = dstnode;
	
	fmemcpy((void far *)dstnode, (void far *)srcnode, sizeof(FNODE));
	
	prev_folder = dstnode;
    }
    
    if (dstnode != (FNODE far *)NULL)
	dstnode->f_next = (FNODE far *)NULL;
    else return(FALSE);

    G.g_ndirs0 = thepath->p_count = --ii;	/* don't count the root */
	
    fstrcpy( &thepath->p_spec[1], ":\\*.*");
	
    return(TRUE);
    
} /* end tree_copy() */

/****************************************************************
 * Open the tree pertaining to the given PNODE.  First make sure
 * the given drive is still valid.  The tree may have been built
 * already, in which case it's either still in memory or has been
 * paged out to disk.
 ****************************************************************/
WORD tree_open( PNODE far * thepath, BOOLEAN end_draw)
{
    BYTE	drv;

    drv = thepath->p_spec[0]-'A';
    dos_sdrv(drv);
    if (DOS_ERR)
    {
	hot_close(G.g_cwin);	/* Recover from error to open a disk */
	return(FALSE);
    }
    else dos_chdir( (LONG)(BYTE far *)"\\" );
	
    G.g_nfiles0 = 0x0L;	
    G.g_size0 = 0x0L;	

    /* Tree is already built?  Just load info on it into the path node.
     * If the path node doesn't already point to this tree, build a copy
     * of the tree in a new FNODE list.
     */
    if (G.t_drive[drv] != (FNODE far *)NULL)
    {
	if (thepath->p_flist != G.t_drive[drv])
	{
	    if (!tree_copy(thepath, G.t_drive[drv], end_draw))
		return(FALSE);
	}
	else G.g_ndirs0 = thepath->p_count = (tree_size(G.t_drive[drv]) - 1);
	return(TRUE);
    }

    fstrcpy( &thepath->p_spec[1], ":\\*.*");
    fl_free(thepath->p_flist);
	
    /* Already built but not showing (paged out).  Page it back in. */
    if (tree_restore(thepath, end_draw))
    {
	G.t_drive[drv] = thepath->p_flist;
	G.g_ndirs0 = thepath->p_count;
	return(TRUE);
    }

    return(tree_build(thepath, drv, end_draw));

} /* end tree_open() */

/****************************************************************
 * Scan the FNODES and build the information needed to print
 * the tree entries.  This fills in the f_treetag and f_tree
 * fields of the tree's FNODEs.  It also fills in the p_count
 * field in the path node with the number of folders (not 
 * including the root) in the tree.
 ****************************************************************/
MLOCAL	void tree_parse( PNODE far * thepath)
{
    UBYTE	column[MAX_LEVEL+1];
    UWORD	n, k;
    FNODE far	*start;
    FNODE far	*ahead;
    UBYTE	siblkids;
    UBYTE	sibling[NUM_TNODES];	/* REVISIT */
					/* NOTE: a byte would overflow if */
				        /* ...a folder had > 256 siblings */
					/* (acceptable for this release)  */

	for (n = 0; n < MAX_LEVEL+1; column[n++]=0);

	folder = thepath->p_flist;
	folder->f_tree[0] = 0;				/* root entry */
	folder->f_treetag = HIDEKIDS;
	
	if (folder->f_next == (FNODE far *)NULL)	/* there's ONLY a root */
	{
	    folder->f_treetag = NOKIDS;
	    thepath->p_count = 0;
	    return;
	}
	else folder = folder->f_next;	/* do the rest on everything BUT root */
	start = folder;
	
	/* Count each node's "siblings".  The count includes all of the node's
	 * children, the node's siblings, and all of the siblings' children,
	 * EXCEPT FOR the children of the LAST of the node's siblings (or it's
	 * own children if it has no siblings).
	 */
	for (n = 0; folder != (FNODE far *)NULL; folder = folder->f_next, n++)
	{
 	  if ((folder->f_next != (FNODE far *)NULL) &&
	      (GET_TLVL(folder->f_next) > GET_TLVL(folder))) /* has child? */
 	      folder->f_treetag = HIDEKIDS;
 	  else folder->f_treetag = NOKIDS;
  	  
	  sibling[n] = 0;
	  siblkids = 0;
	  
	  for (ahead = folder->f_next; 
	       (ahead != (FNODE far *)NULL) && 
	       (GET_TLVL(ahead) >= GET_TLVL(folder));
	       ahead = ahead->f_next)
	  {
	      sibling[n]++;
	      if (GET_TLVL(ahead) == GET_TLVL(folder))	/* sibling: reset who */
		  siblkids = 0;				/* to count kids of.  */
	      else if (GET_TLVL(ahead) > GET_TLVL(folder))
		  siblkids++;
	  }
	  if (sibling[n])
	      sibling[n] -= siblkids;	/* subtract children of LAST sibling */
	}
	thepath->p_count = n;

	/* When a level has siblings we do | or |-, and when it has no
	 * siblings we do |_ or ' '.
	 */
	for (n = 0, folder = start; folder != (FNODE far *)NULL;
		folder = folder->f_next, n++)
	{
	  column[GET_TLVL(folder)] = sibling[n];

	  for (k = 0; (k < MAX_LEVEL+1) && (k <= GET_TLVL(folder)); k++)
	  {
	    if (k == GET_TLVL(folder))
	    {
	      if (column[k]==0)
		folder->f_tree[k]= RIGHT_ELL;
	      else
		folder->f_tree[k]= LEFT_TEE ;
	    }
	    else
	    {
	       if (column[k]==0)
		 folder->f_tree[k]=' ';
	       else
		 folder->f_tree[k]= VERT_BAR ;
	    }
	    if (column[k])
	    	column[k]--;
          }
	  folder->f_tree[k] = 0;
	}
	
} /* end tree_parse() */

/****************************************************************
 *  Build a path spec as it would appear were a folder being opened
 *  eg. C:\
 *      |-FRED
 *      |  |-BILL <-- p_spec = C:\FRED\*.*, f_name = BILL
 ****************************************************************/
void tree_pspec( PNODE far * thepath )
{
BYTE		pathnodes[MAX_LEVEL+1][LEN_ZFNAME+1];
WORD		curr, n; /* currently selected file node's level in tree */
FNODE far *	fn;

	/* Index into selected object */
	for (fn = thepath->p_flist; fn != (FNODE far *)NULL && 
		!(G.g_screen[fn->f_obid].ob_state & SELECTED); 
		fn = fn->f_next)
	{
	    curr = fstrlen(fn->f_tree);
	    fstrcpy((BYTE far *)pathnodes[curr], fn->f_name);
	}
	curr = fstrlen(fn->f_tree); /* Level of current directory */
	
	/* If at the root leave off the *.* */	
	if (curr == 0 || fn == (FNODE far *)NULL)
	{
	    fn=thepath->p_flist;
	    fstrcpy( thepath->p_spec, fn->f_name);
	}
	else
	{
	    /* Now put selected node's parent directory in the p_spec field */
	    thepath->p_spec[0] = NULL;
	    for (n = 0; n < curr; n++)
	    {
		fstrcat( thepath->p_spec, (char far *)pathnodes[n] );
	  
#if DBCS
	    if(!fisequalto(thepath->p_spec,fstrlen(thepath->p_spec)-1,'\\'))
#else /* DBCS */
		if (thepath->p_spec[fstrlen(thepath->p_spec)-1] != '\\')
#endif /* DBCS */
		    fstrcat( thepath->p_spec, "\\");
	    }
	}
	fstrcat( thepath->p_spec, "*.*");
	
} /* end tree_pspec() */

/****************************************************************
 *  Count the directory levels in the given path and compare that
 *  to the maximum depth allowed.  Return TRUE if path is OK, and
 *  FALSE if path is too deep.
 ****************************************************************/
MLOCAL BOOLEAN tree_lvlchk(BYTE *chkpath)
{
    BYTE	path[LEN_ZPATH];
    BYTE	count = 0;
    
    strcpy(path, &chkpath[2]);		/* local copy for altering...	*/
					/* (ignoring drive designator)  */
    if (path[strlen(path)] == '\\')
	path[strlen(path)] = '\0';	/* Remove any final backslash. */
    
    while (path[0] != '\0')
    {
	*strrchr(path, '\\') = '\0';
	count++;
    }
    
    if (count > MAX_LEVEL)
	return(FALSE);
    else return(TRUE);
    
} /* end tree_lvlchk() */

/****************************************************************
 *  See if there's a local tree (in RAM) for the given drive.
 *  If so, find the root/folder node (given by the input path)
 *  in the tree and return a pointer to it.
 *
 *  Assumes that paths that are too deep have already been caught
 *  & ignored at a higher level.
 ****************************************************************/
MLOCAL FNODE far * tree_findlocal( BYTE far * thepath )
{
    FNODE far	*dir;
    BYTE	nxtdir[LEN_ZFNAME];
    
    if ((dir = G.t_drive[thepath[0]-'A']) == (FNODE far *)NULL)
	return(dir);
    
    /* NOTE: both root and children of root have level = 0. */
    for (level = 0, thepath = extract(thepath, nxtdir); *nxtdir;
    	    level++, thepath = extract(thepath, nxtdir))
    {
	/* look through FNODEs until we match this subdir */
	while ( (dir != (FNODE far *)NULL) &&
		(fstrcmp( dir->f_name, (BYTE far *)nxtdir) != 0) ||
	        (level != (level ? GET_TLVL(dir)+1 : GET_TLVL(dir))) )
	{
	    dir = dir->f_next;
	}
	/* Match.  Unless that was the last folder to find,
	 * start looking for the next folder.
	 */
	if ( *thepath != NULL )
	    dir = dir->f_next;
	else break;
    }
    
    return(dir);
    
} /* end tree_findlocal() */

/****************************************************************
 *  Find the saved tree node that corresponds to the given full
 *  path string, and move the file seek pointer to point to it.
 *  The given drive's tree is saved in an already opened file,
 *  given by global thndl.
 *     
 *  Don't assume that the file's seek pointer is at 1st byte on
 *  entry.  The requested path is assumed to exist in the tree
 *  (because the tree specifically always reflects the state of
 *   folders on the drive).
 *     
 *  Return the new file seek pointer position if the node is
 *  found, and FALSE otherwise (which only happens if there's a
 *  file read error).
 ****************************************************************/
MLOCAL LONG tree_finddisk( BYTE far * thepath )
{
    WORD	node_count;
    WORD	listsize;
    WORD	idx, level;
    BYTE	nxtdir[LEN_ZFNAME];
    
    dos_lseek(thndl, SEEK_SET, 0L);
    dos_read(thndl, sizeof(WORD), (LONG)(WORD far *)&node_count );
    listsize = node_count * sizeof(FNODE);

    if (!tree_thaw(&listsize))
	return(FALSE);				/* file read error */

    /* Read in nodes and search them, a bufferfull at a time. */
    /* NOTE: children of root have level = 0. */
    for (idx = level = 0, thepath = extract( thepath, (BYTE far *)nxtdir); *nxtdir;
    	    level++, thepath = extract( thepath, (BYTE far *)nxtdir))
    {
	/* Look through buffered FNODEs until we match this subdir.  FNODEs
	 * that have been deleted from the file will be skipped by this test.
	 */
	while ( (idx == TBUF_LEN - 1) ||
		(strcmp( tbuffer[idx].f_name, nxtdir) != 0) ||
	        (level != (level ? GET_TLVL(&tbuffer[idx])+1 : 
				   GET_TLVL(&tbuffer[idx]))) )
	{
	    /* Refill FNODE buffer when necessary. */
	    if (idx == TBUF_LEN - 1)
	    {
		if (!tree_thaw(&listsize))
		    return(FALSE);		/* file read error */
		idx = 0;
	    }
	    else idx++;
	}
	/* Match.  Unless that was the last folder to find,
	 * start looking for the next folder.
	 */
	if ( *thepath != NULL )	/* more to find */
	    idx++;
	else break;
    }
    
    /* Move file seek pointer to 1st byte of the found FNODE and return 
     * that position.
     */
    return(dos_lseek(thndl, SEEK_SET,
			  (LONG)(sizeof(WORD) + (idx * sizeof(FNODE)))));

} /* end tree_finddisk() */

/****************************************************************
 *  If there's a saved tree for the given drive, find the node
 *  that corresponds to the given path and update the folder's
 *  name.
 ****************************************************************/
void tree_rename( BYTE *thepath, BYTE far *newname )
{
    FNODE far	*thenode;

    /* Ignore a path that's too many levels deep to be in a tree at all. */
    if (!tree_lvlchk(thepath))
	return;

    /* First look for a local version of the tree to update. */
    thenode = tree_findlocal( (char far *)thepath ) ;
    if ( thenode != (FNODE far *)NULL )
	fstrcpy(thenode->f_name, newname);
    
    else if (tree_issaved(thepath[0]))
    {
	if (tree_finddisk( (BYTE far *)thepath) != 0L)
	{	
	    /* File seek ptr is at FNODE to change.  Move it to name field. */
	    dos_lseek(thndl, SEEK_CUR, 14L);
	    dos_write(thndl, fstrlen(newname) + 1, (LONG)(BYTE far *)newname );
	}

	dos_close(thndl);	
    }

} /* end tree_rename() */

/****************************************************************
 *  Subroutine to fill in and modify the contents of nodes when a
 *  new node is added in between its parent and its
 *  next nodes.  Parent node's treetag may need to be changed.
 *  The new node's f_name, f_treetag, f_tree[] and level fields
 *  must be filled in based on its parent and next nodes.
 *  Return TRUE if the prnt node changed, FALSE otherwise.
 ****************************************************************/
MLOCAL BOOLEAN tree_subadd(FNODE far *prnt, FNODE far *new, FNODE far *next, 
			 BYTE *newfold)
{
    BYTE far	*theone;
    BOOLEAN	prntmod;

    if (prnt->f_treetag == NOKIDS)	/* parent that HAD no kids... */
    {					/* ....now has a visible kid  */
	prnt->f_treetag = HIDEKIDS;
	prntmod = TRUE;
    }
    else prntmod = FALSE;
    
    /* Fill in the new FNODE. */
    new->f_attr = F_SUBDIR;
    new->f_size = 0x0L;
    new->f_treetag = NOKIDS;
    fstrcpy(new->f_name, (BYTE far *)newfold);
    fstrcpy(new->f_tree, prnt->f_tree);
    SET_ISTREE(new, TRUE);
	
    /* Unless the new folder is a child of the root, put | over the (one)
     * instance of |- or |_ from the parent's tree[].
     */
    if (new->f_tree[0] != '\0')
    {
	if ((theone = fstrchr(new->f_tree, LEFT_TEE)) != (BYTE far *)NULL)
	    *theone = VERT_BAR;
	else
	{
	    theone = fstrchr(new->f_tree, RIGHT_ELL);
	    *theone = ' ';
	}
	SET_TLVL(new, (GET_TLVL(prnt) + 1));
    }
    else SET_TLVL(new, 0);		/* children of root have.... */
					/* ....same level as root    */

    /* Put the final branch on the new node.
     * If it has siblings, use |- and if it doesn't, use |_ 
     */
    theone = new->f_tree + fstrlen(new->f_tree);
    *theone = ((next != (FNODE far *)NULL) &&
	       (GET_TLVL(next) == GET_TLVL(new))) ? LEFT_TEE : RIGHT_ELL;

    *(theone+1) = '\0';

    return(prntmod);
	
} /* end tree_subadd() */

/****************************************************************
 *  Dynamically update any affected tree when a new folder has
 *  been added, as in the given path.  There seems to be no sort
 *  happening for tree lists, so for now just add the new folder
 *  in as the first child of its parent.
 *  Don't bother to check for adding below the deepest level.
 *  That will have been taken care of before we get here.
 ****************************************************************/
void tree_adddisk(BYTE *prntpath, BYTE *newfold)
{
    FNODE	prntnode, newnode, nextnode;
    FNODE far	*pf_next;
    LONG	node_at, file_end, move_from;
    WORD	move_amt, move_ttl;
    WORD	node_count;
    LONG	total, avail;
    BYTE	ii;    
    
    if (!tree_issaved(prntpath[0]))		/* tree saved to disk */
	return;

    dos_read(thndl, sizeof(WORD), (LONG)(WORD far *)&node_count );
    node_count++;

    /* Make sure there's enough disk space for the node to be added. */
    dos_space(start_path[0] - 'A' + 1, &total, &avail);
    if (avail < sizeof(FNODE))
    {
	dos_close(thndl);
	dos_delete( (LONG)(BYTE far *)treefile );	/* don't trust file contents! */
	return;		
    }
	
    if ((node_at = tree_finddisk( (BYTE far *)prntpath)) == 0L)
    {
	dos_close(thndl);
	return;
    }

    dos_read(thndl, sizeof(FNODE), (LONG)(FNODE far *)&prntnode );

    /* Find the next non-empty node after the parent (if there is one). */
    pf_next = (FNODE far *)NULL;
    file_end = dos_lseek(thndl, SEEK_END, 0L);
    dos_lseek(thndl, SEEK_SET, node_at + sizeof(FNODE));
    while (dos_lseek(thndl, SEEK_CUR, 0L) < file_end)
    {
	dos_read(thndl, sizeof(FNODE), (LONG)(FNODE far *)&nextnode );
	if (nextnode.f_name[0] != '\0')
	{
	    pf_next = (FNODE far *)&nextnode;
	    break;
	}
    }

    /* Build the new node and any changes to the parent node, and
     * write out the new parent node if it was changed. 
     */
    if (tree_subadd((FNODE far *)&prntnode, (FNODE far *)&newnode, 
							pf_next, newfold))
    {
	dos_lseek(thndl, SEEK_SET, node_at);
	dos_write(thndl, sizeof(FNODE), (LONG)(FNODE far *)&prntnode );
    }
	    
    node_at += sizeof(FNODE);

    /* If necessary, make a space in the file for the new node by
     * moving everything else forward one node.  Otherwise, the
     * new node goes on the end of the file.
     */
    if (node_at < file_end)
    {
	move_ttl = (WORD)(file_end - node_at);
	
	for (ii = 1; move_ttl; move_ttl -= move_amt, ii++)
	{
	    if (move_ttl < TBUF_LEN)
	    {
		move_amt = move_ttl;
		move_from = node_at;
	    }
	    else
	    {
		move_amt = TBUF_LEN;
		move_from = (LONG)(file_end - (ii * move_amt));
	    }
	    dos_lseek(thndl, SEEK_SET, move_from);
	    dos_read(thndl, move_amt, (LONG)(BYTE far *)tbuffer );
	    dos_lseek(thndl, SEEK_SET, move_from + sizeof(FNODE));
	    dos_write(thndl, move_amt, (LONG)(BYTE far *)tbuffer );
	}
    }

    /* Write out the new node and node count. */
    dos_lseek(thndl, SEEK_SET, node_at);
    dos_write(thndl, sizeof(FNODE), (LONG)(FNODE far *)&newnode );
    dos_lseek(thndl, SEEK_SET, 0L);
    dos_write(thndl, sizeof(WORD), (LONG)(WORD far *)&node_count );

    dos_close(thndl);
    
} /* end tree_adddisk() */
	
/****************************************************************
 *  Dynamically update any affected tree when a new folder has
 *  been added, as in the given path.  There seems to be no sort
 *  happening for tree lists, so for now just add the new folder
 *  in as the first child of its parent.
 *  Don't bother to check for adding below the deepest level.
 *  That will have been taken care of before we get here.
 ****************************************************************/
void tree_add(BYTE *nodepath)
{
    BYTE	prntpath[LEN_ZPATH];
    BYTE	newfold[LEN_ZFNAME];
    BYTE	*lastslsh;
    FNODE far	*pf_prnt;
    FNODE far	*pf_new;
    BYTE	ii;    
    
    /* Ignore a path that's too many levels deep to be in a tree at all. */
    if (!tree_lvlchk(nodepath))
	return;

    strcpy(prntpath, nodepath);			/* local copy for altering */
    lastslsh = strrchr(prntpath, '\\');
    strcpy(newfold, lastslsh + 1);
    *lastslsh = '\0';

    pf_prnt = tree_findlocal( (char far *)prntpath ) ;
    if ( pf_prnt != (FNODE far *)NULL )
    {
	/* Allocate the new FNODE and link it in. */
	if ((pf_new = fn_alloc()) == (FNODE far *)NULL)
	{
	    alert(1, EROBJMIS);			/* no room to add node, so... */
	    tree_dirty(nodepath[0]-'A');	/* ...discard incomplete tree */
	    return;
	}
	pf_new->f_next = pf_prnt->f_next;
	pf_prnt->f_next = pf_new;

	tree_subadd(pf_prnt, pf_new, pf_new->f_next, newfold);

	/* Update any TREEWIN that's showing the destination drive. */
	for (ii = 2; ii < 4; ii++)
	{
	    if ( (G.g_wlist[ii].w_open) && 
		 (G.g_wlist[ii].w_path->p_spec[0] == prntpath[0]) )
	        G.g_ndirs0 = ++(G.g_wlist[ii].w_path->p_count);
	}

    } /* end if: local tree */
	
    else tree_adddisk(prntpath, newfold);
        
} /* end tree_add() */

/****************************************************************
 *  Subroutine to modify the contents of nodes affected by the 
 *  deletion of a node.  All nodes between and including the
 *  deleted node's parent and the deleted node may need revision.
 *
 *  NOTE that because the root and the root's immediate children
 *  both have LEVEL = 0 (where level is packed into high nibble of 
 *  FNODE.f_istree), we have to use the length the FNODE.f_tree
 *  string instead of GET_TLVL() in many cases to differentiate 
 *  between them.
 ****************************************************************/
MLOCAL void tree_subdel(FNODE far *prnt, FNODE far *prev, FNODE far *del,
			 FNODE far *next)
{
    WORD	siblvl, branch;
    FNODE far	*pf;

    /* If prev now has no kids then get rid of its treetag */
    if ( !((next != (FNODE far *)NULL) &&
	  (fstrlen(next->f_tree) > fstrlen(prev->f_tree))) )
	prev->f_treetag = NOKIDS;

    /*	     |--PRNT
     *	     |	|--AAA		Get rid of the branch pointed to by the      
     *       |->|  |---BBB	arrows in the case shown.  Find the last child
     *       |->|  |---CCC	of the parent and erase the obsolete branch line
     *       |->|  |___DDD	on all of its children (and grandchildren).
     *       |->|      |___PREV
     *       |->|
     *       |	|__DELETED.ONE
     *	     |
     *	     |--ETC	
     *
     * If there's a previous sibling that has children, and there's no next
     * sibling, then there's an obsolete branch to delete.
     */             
    if ( (GET_TLVL(prev) > GET_TLVL(del)) &&
	 !((next != (FNODE far *)NULL) && (GET_TLVL(next) == GET_TLVL(del))) )
    {
	pf = prnt->f_next;			/* start at 1st sibling */
	siblvl = fstrlen(pf->f_tree);		/* remember its level */
	for ( ; (pf->f_next != (FNODE far *)NULL) &&
		(fstrlen(pf->f_next->f_tree) >= siblvl); pf = pf->f_next)
	{
	    if (fstrlen(pf->f_tree) == siblvl)
		prnt = pf;
	}

	/* PRNT now points to last immediate child of the original parent.
	 * Walk through everything under it and erase the right branch.
	 */
	branch = siblvl - 1;
	siblvl = GET_TLVL(prnt);		/* remember this */
	for (pf = prnt->f_next;
		(pf != (FNODE far *)NULL) && (GET_TLVL(pf) > siblvl);
		pf = pf->f_next)
	    pf->f_tree[branch] = ' ';
    }
    /* If prev is no longer followed by a sibling then it gets the |_ 
     * unless prev is the root node.
     */
    else if ((prev->f_next == (FNODE far *)NULL) ||
	     (fstrlen(prev->f_tree) > fstrlen(prev->f_next->f_tree)))
    {
	if (fstrlen(prev->f_tree))
	    prev->f_tree[fstrlen(prev->f_tree)-1] = RIGHT_ELL;
    }
    
} /* end tree_subdel() */

/****************************************************************
 *  Dynamically update any affected tree when a folder has been
 *  deleted, as in the given path.
 *
 *  A node deleted from a tree saved in a disk file will leave an
 *  empty hole.  Holes will be ignored on subsequent parsing of
 *  the file's contents.
 *  NOTE: this means that file's node count doesn't change when
 *  a node is "deleted" (emptied).
 ****************************************************************/
void tree_deldisk(BYTE *delpath)
{
    BYTE	delfold[LEN_ZFNAME];
    BYTE	*lastslsh;
    LONG	node_at, file_end;
    WORD	readsize, siblvl, idx, next, del, prev;
    BOOLEAN	done;
	
    if (!tree_issaved(delpath[0]))		/* tree saved to disk */
	return;
	
    /* Seperate delete path into parent's path and delete folder name. */
    lastslsh = strrchr(delpath, '\\');
    strcpy(delfold, lastslsh + 1);
    *lastslsh = '\0';	

    /* Find the parent (PRNT) of the deleted node and read nodes into
     * tbuffer[] from PRNT on, with PRNT going into position [0].
     */
    if ((node_at = tree_finddisk( (BYTE far *)delpath)) == 0L)
    {
	dos_close(thndl);
	return;
    }

    file_end = dos_lseek(thndl, SEEK_END, 0L);
    dos_lseek(thndl, SEEK_SET, node_at);
    readsize = (WORD)(file_end - node_at);
    if ((readsize = tree_thaw(&readsize)) == 0)
    {
	dos_close(thndl);
	return;					/* file read error */
    }

    /* Simulate the linking of all nodes (by their f_next pointers)
     * from the PRNT to the node following the deleted node.  "Unlink"
     * the deleted node from this chain.  Skip any previously deleted
     * FNODEs by looking for an empty f_name[] field.
     */
    siblvl = strlen(tbuffer[0].f_tree) + 1;	/* (prnt is NOT a null node) */
    idx = 0;
    done = FALSE;
    while (!done)
    {
	next = idx + 1;
	while (tbuffer[next].f_name[0] == '\0')
	    next++;
	if ( (strlen(tbuffer[next].f_tree) == siblvl) &&
	     (strcmp(tbuffer[next].f_name, delfold) == 0) )
	{
	    done = TRUE;
	    prev = idx;
	    del = next++;
	    while (tbuffer[next].f_name[0] == '\0')
		next++;
	}

	if (next == readsize)
	    tbuffer[idx].f_next = (FNODE far *)NULL;
	else tbuffer[idx].f_next = (FNODE far *)&tbuffer[next];
	idx = next;
    }
	    
    tree_subdel((FNODE far *)tbuffer, (FNODE far *)&tbuffer[prev],
		(FNODE far *)&tbuffer[del], (FNODE far *)&tbuffer[prev].f_next);

    memset(&tbuffer[del], 0, sizeof(FNODE));
		
    /* Write modified nodes and empty deleted node back out to file. */
    dos_lseek(thndl, SEEK_SET, node_at);
    dos_write(thndl, (del+1) * sizeof(FNODE), (LONG)(BYTE far *)tbuffer );
	
    dos_close(thndl);
    
} /* end tree_deldisk() */
    
/****************************************************************
 *  Dynamically update any affected tree when a folder has been
 *  deleted, as in the given path.
 *
 *  First look for the drive's tree in memory.  If it exists both
 *  in memory and on disk, the memory version will always take
 *  precedence.
 ****************************************************************/
void tree_delete(BYTE *nodepath)
{
    BYTE	delpath[LEN_ZPATH], ii;
    FNODE far	*pf_prnt;
    FNODE far	*pf_prev;
    FNODE far	*pf_del;
        
    /* Ignore a path that's too many levels deep to be in a tree at all. */
    if (!tree_lvlchk(nodepath))
	return;

    strcpy(delpath, nodepath);			/* local copy for altering */
    
    pf_del = tree_findlocal( (char far *)delpath ) ;
    if ( pf_del != (FNODE far *)NULL )
    {
	*(strrchr(delpath, '\\')) = '\0';
	pf_prnt = tree_findlocal( (char far *)delpath );
	
	/* find the immediate predecessor of the node to delete */
	for (pf_prev = pf_prnt; pf_prev->f_next != pf_del; 
						pf_prev = pf_prev->f_next);
	
	pf_prev->f_next = pf_del->f_next;	/* unlink the deleted kid */

	tree_subdel(pf_prnt, pf_prev, pf_del, pf_prev->f_next);

	fn_free(pf_del);

	/* Update any TREEWIN that's showing the destination drive. */
	for (ii = 2; ii < 4; ii++)
	{
	    if ( (G.g_wlist[ii].w_open) && 
		 (G.g_wlist[ii].w_path->p_spec[0] == delpath[0]) )
	        G.g_ndirs0 = --(G.g_wlist[ii].w_path->p_count);
	}
    }    
    else tree_deldisk(delpath); 
    
} /* end tree_delete() */



/*
 *	EOF:	desktree.c
 */
